/*
 * alpha.hpp
 *
 */
#ifndef ALPHA_HPP
#define ALPHA_HPP

#include "util.hpp"

namespace RFOLD {
class Alpha {
public:
  typedef char CodeT;
  typedef Array<CodeT> Codes;
  enum CodeType {
    NCODE_TYPE,
    PCODE_TYPE,
    SCODE_TYPE,
    BAD_CODE_TYPE
  };
  enum {
    N_CODE_TYPE = (SCODE_TYPE - NCODE_TYPE + 1),
    N_CODE_TYPE0 = (BAD_CODE_TYPE - NCODE_TYPE + 1)
  };
  typedef CArray<string, N_CODE_TYPE0> CodeTypeToStrs;
  static const CodeTypeToStrs CODE_TYPE_TO_STRS;
  static const string code_type_to_str(int c) {return CODE_TYPE_TO_STRS[c];}
  static int str_to_code_type(const string& str) {
    static const CArray<int, N_CODE_TYPE0> code_types = {{
      NCODE_TYPE, PCODE_TYPE, SCODE_TYPE, BAD_CODE_TYPE
    }};
    for (int i = 0; i < (int)CODE_TYPE_TO_STRS.size(); i++) {
      if (str == CODE_TYPE_TO_STRS[i]) {
	return code_types[i];
      }
    }
    return BAD_CODE_TYPE;
  }
  enum NCode {
    N_A,
    N_C,
    N_G,
    N_U,
    N_N,
    N_GAP,
    N_BAD
  };
  enum {
    NNCODE   = (N_U - N_A + 1),
    NNCODEN  = (N_N - N_A + 1),
    NNCODENG = (N_GAP - N_A + 1),
    NNCODE0  = (N_BAD - N_A + 1)
  };
  typedef CArray<CodeT, 256> CharToNCode;
  typedef CArray<char, NNCODE0> NCodeToChar;
  typedef CArray<string, NNCODE0> NCodeToStrs;

  static const CharToNCode CHAR_TO_NCODE;
  static const NCodeToChar NCODE_TO_CHAR;
  static const NCodeToStrs NCODE_TO_STRS;
  static const string BASE_CHARS;
  static const string GAP_CHARS;
  static const string AMB_CHARS;

  static CodeT ncode(char c) {return CHAR_TO_NCODE[(unsigned char)c];}
  static string ncode_to_str(CodeT c) {return NCODE_TO_STRS[c];}
  static bool is_base_char(char c) {return is_base_ncode(ncode(c));}
  static bool is_gap_char(char c) {return is_gap_ncode(ncode(c));}
  static bool is_amb_char(char c) {return is_amb_ncode(ncode(c));}
  static bool is_base_ncode(CodeT c) {return (c < N_N);}
  static bool is_gap_ncode(CodeT c) {return (c == N_GAP);}
  static bool is_amb_ncode(CodeT c) {return (c == N_N);}
  static bool is_bad_ncode(CodeT c) {return (c == N_BAD);}
  static bool contains_bad_ncode(const Codes& codes) {
    for (int i = 0; i < (int)codes.size(); i++) {
      if (is_bad_ncode(codes[i])) return true;
    }
    return false;
  }
  template <typename InpIt>
  static Codes str_to_ncodes(InpIt first, InpIt last) {
    Codes codes(std::distance(first, last));
    for (int i = 0; i < (int)codes.size(); i++) {
      codes[i] = CHAR_TO_NCODE[*first++];
    }
    Assert(!contains_bad_ncode(codes));
    return codes;
  }
  template <typename InpIt>
  static Array<char> ncodes_to_str(InpIt first, InpIt last) {
    Array<char> str(std::distance(first, last));
    for (int i = 0; i < (int)str.size(); i++) {
      str[i] = NCODE_TO_CHAR[*first++];
    }
    return str;
  }
  static Array<char> replace_amb_chars_with_rand_bases(const Array<char>& str) {
    Array<char> ret(str);
    for (int i = 0; i < (int)ret.size(); i++) {
      if (is_amb_char(ret[i])) {
	ret[i] = BASE_CHARS[Rand(BASE_CHARS.size())];
      }
    }
    return ret;
  }
  static Codes replace_amb_ncodes_with_rand_bases(const Codes& codes) {
    static const CArray<CodeT, NNCODE> ncodes = {{N_A, N_C, N_G, N_U}};
    Codes ret(codes);
    for (int i = 0; i < (int)ret.size(); i++) {
      if (is_amb_ncode(ret[i])) {
	ret[i] = ncodes[Rand(ncodes.size())];
      }
    }
    return ret;
  }
  static Array<char> generage_rand_rna(int len) {
    Array<char> s(len);
    for (int i = 0; i < len; i++) {
      s[i] = BASE_CHARS[Rand(BASE_CHARS.size())];
    }
    return s;
  }

  enum PCode {
    P_AA,
    P_AC,
    P_AG,
    P_AU,
    P_CA,
    P_CC,
    P_CG,
    P_CU,
    P_GA,
    P_GC,
    P_GG,
    P_GU,
    P_UA,
    P_UC,
    P_UG,
    P_UU,
    P_NN,
    P_GAP,
    P_BAD
  };
  enum {
    NPCODE   = (P_UU - P_AA + 1),
    NPCODEN  = (P_NN - P_AA + 1),
    NPCODENG = (P_GAP - P_AA + 1),
    NPCODE0  = (P_BAD - P_AA + 1)
  };
  typedef CArray<CodeT, 2> NCode2;
  typedef CArray<string, NPCODE0> PCodeToStrs;
  typedef CArray<CArray<CodeT, NNCODE0>, NNCODE0> NCode2ToPCode;
  typedef CArray<NCode2, NPCODE0> PCodeToNCode2;
  typedef CArray<CodeT, NPCODE0> FlipPCode;

  static const PCodeToStrs PCODE_TO_STRS;
  static const NCode2ToPCode NCODE2_TO_PCODE;
  static const PCodeToNCode2 PCODE_TO_NCODE2;
  static const FlipPCode FLIP_PCODE;

  enum SCode {
    S_AU,
    S_CG,
    S_GC,
    S_GU,
    S_UA,
    S_UG,
    S_MM,
    S_NN,
    S_GAP,
    S_BAD
  };
  enum {
    NSCODE   = (S_MM - S_AU + 1),
    NSCODEN  = (S_NN - S_AU + 1),
    NSCODENG = (S_GAP - S_AU + 1),
    NSCODE0  = (S_BAD - S_AU + 1),
    N_CANONICAL_SCODE = (S_UG - S_AU + 1)
  };
  typedef CArray<string, NSCODE0> SCodeToStrs;
  typedef CArray<CArray<CodeT, NNCODE0>, NNCODE0> NCode2ToSCode;
  typedef CArray<NCode2, NSCODE0> SCodeToNCode2;
  typedef CArray<CodeT, NSCODE0> FlipSCode; 
  typedef CArray<CodeT, NPCODE0> PCodeToSCode;

  static const SCodeToStrs SCODE_TO_STRS;
  static const NCode2ToSCode NCODE2_TO_SCODE;
  static const SCodeToNCode2 SCODE_TO_NCODE2;
  static const FlipSCode FLIP_SCODE;
  static const PCodeToSCode PCODE_TO_SCODE;

  static CodeT pcode(CodeT c, CodeT c1) {return NCODE2_TO_PCODE[c][c1];}
  static string pcode_to_str(CodeT c) {return PCODE_TO_STRS[c];}
  static std::pair<CodeT, CodeT> split_pcode(CodeT c) {
    const NCode2& a = PCODE_TO_NCODE2[c]; return std::make_pair(a[0], a[1]);}
  static CodeT flip_pcode(CodeT c) {return FLIP_PCODE[c];}
  static CodeT scode(CodeT c, CodeT c1) {return NCODE2_TO_SCODE[c][c1];}
  static string scode_to_str(CodeT c) {return SCODE_TO_STRS[c];}
  static std::pair<CodeT, CodeT> split_scode(CodeT c) {
    const NCode2& a = SCODE_TO_NCODE2[c]; return std::make_pair(a[0], a[1]);}
  static CodeT flip_scode(CodeT c) {return FLIP_SCODE[c];}
  static CodeT pcode_to_scode(CodeT c) {return PCODE_TO_SCODE[c];}
  static bool is_canonical_scode(CodeT c) {return (c < S_MM);}
  static bool is_canonical_pcode(CodeT c) {return is_canonical_scode(pcode_to_scode(c));}
  static bool is_canonical(CodeT c, CodeT c1) {return is_canonical_scode(scode(c, c1));}
  static bool is_canonical_chars(char c, char c1) {return is_canonical(ncode(c), ncode(c1));}
};
}
#endif
