/*
 * array_file.hpp
 */
#ifndef ARRAY_FILE
#define ARRAY_FILE

#include <fstream>

namespace RFOLD {
template<typename T>
class ArrayFile0 {
public:
  typedef T              value_type;
  typedef T*             iterator;
  typedef const T*       const_iterator;
  typedef T&             reference;
  typedef const T&       const_reference;
  typedef std::size_t    size_type;
  typedef std::ptrdiff_t difference_type;

  enum {
    UNIT0 = 8,
    OFFSET0 = (UNIT0 * 2),
    UNIT = sizeof(T)
  };
  ArrayFile0()
    : _size(0) {}
  ArrayFile0(const std::string& filename) 
    : _size(0) {set_file(filename);}
  ~ArrayFile0() {if (_fh.is_open()) {_fh.close();}}
  void set_file(const std::string& filename) {
    if (_fh.is_open()) {_fh.close();}
    _filename = filename;
    _size = 0;
    _fh.open(_filename.c_str(), (std::ios::in | std::ios::out | std::ios:: trunc));
    Check(_fh, "cannot open file %s", _filename.c_str());
    for (int i = 0; i < OFFSET0; i++) {_fh.put(0);}
  }
  std::string filename() const {return _filename;}
  bool empty() const {return (_size == 0);}
  size_type size() const {return _size;}
  void clear() {if (_fh.is_open()) _fh.seekp(OFFSET0); _size = 0;}
  void resize(size_type size, const T& val = T()) {
    if (_size < size) {
      fill_range(_size, size, val);
    } else {
      _size = size;
    }
  }
  void fill_range(size_type b, size_type e, const_reference val) {// includes resizing
    Assert(0 <= b && b <= e);
    size_type n = (e - b);
    if (n == 0) return;

    size_type offset = get_offset(b);
    Check(_fh.is_open());
    _fh.seekp(offset);
    while (n-- > 0) {
      _fh.write((char const*)&val, UNIT);
    }
    if (_size < e) {
      _size = e;
    }
  }
  void fill(const T& val) {fill_range(0, _size, val);}
  void operator<<(const_reference val) {fill_range(_size, (_size+1), val);}
  void read_range(size_type b, size_type e, T* t) {
    size_type n = (e - b) * UNIT;
    if (n == 0) return;
    Assert(0 <= b && b <= e && e <= _size);
    size_type offset = get_offset(b);
    Check(_fh.is_open());
    _fh.seekg(offset);
    _fh.read((char*)t, n);
  }
  void write_range(size_type b, size_type e, T const* t, const_reference t1 = T()) {
    Assert(0 <= b && b <= e);
    size_type n = (e - b) * UNIT;
    if (n == 0) return;

    if (_size <= b) {// fill gap region. changes _size
      fill_range(_size, b, t1);
    }
    size_type offset = get_offset(b);
    Check(_fh.is_open());
    _fh.seekp(offset);
    _fh.write((char const*)t, n);
    if (_size < e) {
      _size = e;
    }
  }
  
private:
  fstream _fh;
  size_type _size;
  string _filename;

  size_type get_offset(size_type n) const {return (OFFSET0 + n * UNIT);}
};
}

namespace RFOLD {
// simple array
template<typename T>
class ArrayFile1 {
public:
  typedef Array<T> Data;
  typedef T              value_type;
  typedef T*             iterator;
  typedef const T*       const_iterator;
  typedef T&             reference;
  typedef const T&       const_reference;
  typedef std::size_t    size_type;
  typedef std::ptrdiff_t difference_type;

  enum {
    FACTOR = 2
  };
  ArrayFile1() : _size(0) {}
  ArrayFile1(const std::string& filename) 
    : _size(0) {set_file(filename);}
  void set_file(const std::string& filename) {_filename = filename;}
  std::string filename() const {return _filename;}
  bool empty() const {return (_size == 0);}
  size_type size() const {return _size;}
  void clear() {_data.clear(); _size = 0;}
  void resize(size_type size, const T& val = T()) {
    if (_size < size) {
      fill_range(_size, size, val);
    } else {
      _size = size;
    }
  }
  void fill_range(size_type b, size_type e, const_reference val) {// includes resizing
    Assert(0 <= b && b <= e);
    if (b == e) return;

    if (_size < e) {
      _data.resize(FACTOR * e);
      _size = e;
    }
    for (size_type i = b; i < e; i++) {
      _data[i] = val;
    }
  }
  void fill(const T& val) {fill_range(0, _size, val);}
  void operator<<(const T& val) {fill_range(_size, (_size+1), val);}
  void read_range(size_type b, size_type e, iterator t) {
    Assert(0 <= b && b <= e && e <= _size);
    for (size_type i = b; i < e; i++) {
      (*t++) = _data[i];
    }
  }
  void write_range(size_type b, size_type e, const_iterator t, const_reference t1 = T()) {
    Assert(0 <= b && b <= e);
    if (b == e) return;

    size_type n = size();
    if (n < e) {
      _data.resize(e);
      _size = e;
    }
    if (n <= b) {// fill gap region.
      fill_range(n, b, t1);
    }
    for (size_type i = b; i < e; i++) {
      _data[i] = (*t++);
    }
  }
  
private:
  Data _data;
  size_type _size;
  string _filename;
};
}

#include "array_range.hpp"

namespace RFOLD {
template <typename T>
class ArrayFile {
public:
#if USE_ARRAY_FILE
  typedef ArrayFile0<T> Data;
#else
  typedef ArrayFile1<T> Data;
#endif
  typedef ArrayRange<T> Cache;

  typedef T              value_type;
  typedef T*             iterator;
  typedef const T*       const_iterator;
  typedef T&             reference;
  typedef const T&       const_reference;
  typedef std::size_t    size_type;
  typedef std::ptrdiff_t difference_type;

  ArrayFile() : _range_size(0), _margin(0), _default(T()), _range_begin(0), _range_end(0) {
    setup_array_range();
  }
  ArrayFile(const string& filename, difference_type range_size, difference_type margin,
	    const_reference value)
    : _array_file(filename), _range_size(range_size), _margin(margin), 
      _default(value), _range_begin(0), _range_end(0) {
    setup_array_range();
  }
  bool empty() const {return (_range_end == _range_begin) &&_array_file.empty();}
  difference_type size() const {return std::max(_range_end, (difference_type)_array_file.size());}
  void resize(difference_type n) {flush(); _array_file.resize(n, _default);}
  void clear() {
    _array_file.clear();
    _range_begin = 0;
    _range_end = 0;
    setup_array_range();
  }
  void fill(const_reference value) {
    flush();
    _array_file.fill(value);
    for (difference_type i = _range_begin; i < _range_end; i++) {
      _array_range[i] = value;
    }
  }
  void reset(const std::string& filename, difference_type range_size,
	     difference_type margin, const_reference val) {
    flush();
    _array_file.set_file(filename);
    _range_size = range_size;
    _margin = margin;
    _default = val;
    _range_begin = 0;
    _range_end = 0;
    setup_array_range();
  }
  void set_file(const std::string& filename) {
    flush();
    _array_file.set_file(filename);
    _range_begin = 0;
    _range_end = 0;
    setup_array_range();
  }
  void set_margin(difference_type margin) {
    flush();
    _margin = margin;
    _range_begin = 0;
    _range_end = 0;
    setup_array_range();
  }
  void set_range_size(difference_type range_size) {
    flush();
    _range_size = range_size;
    _range_begin = 0;
    _range_end = 0;
    setup_array_range();
  }
  void set_default(const_reference t) {
    flush();
    _default = t;
    _range_begin = 0;
    _range_end = 0;
    setup_array_range();
  }
  difference_type margin() const {return _margin;}
  difference_type range_size() const {return _range_size;}
  reference operator[](difference_type i) {return _array_range[i];}
  const_reference operator[](difference_type i) const {return _array_range[i];}
  void push_back(const T& t) {
    difference_type n = size();
    load_range_if_needed(n, n+1);
    _array_range[n] = t;
  }
  void get_range(difference_type& b, difference_type& e) {
    b = _range_begin, e = _range_end;}
  difference_type range_begin() const {return _range_begin;}
  difference_type range_end() const {return _range_end;}
  void load_range_if_needed(difference_type b, difference_type e) {
    Assert(0 <= b && b <= e);
    if (b < _range_begin) {
      if ((_array_range.index_begin()+_margin) <= b) {
	_range_begin = b;
      } else {
	load_range(std::max((difference_type)0, (difference_type)(e-_range_size)), e);
      }
    } else if (_range_end < e) {
      if (e <= (_array_range.index_end()-_margin)) {
	_range_end = e;// includes resizing
      } else {// includes resizing
	difference_type new_size = std::max(e, size());
	load_range(b, std::min(b+_range_size, new_size));
      }
    }
  }
  void load_range(difference_type b, difference_type e) {// includes resizing
    Assert(0 <= b && b <= e);
    flush();
    if (_range_size < (e - b)) {// modify _range_begin, _range_end
      set_range_size(2 * (e - b));
    }
    _range_begin = b;
    _range_end   = e;
    difference_type margin0 = ((difference_type)_array_range.size() - (e - b)) / 2;
    Assert(_margin <= margin0);
    difference_type b0 = std::max((0-_margin), (b-margin0));
    difference_type e0 = (b0 + (difference_type)_array_range.size());
    difference_type b1 = std::max((difference_type)0, (difference_type)b0);
    difference_type e1 = std::max(b1, std::min((difference_type)e0, 
					       (difference_type)_array_file.size()));
    _array_range.set_offset(b0);
    for (difference_type i = b0; i < b1; i++) {
      _array_range[i] = _default;
    }
    _array_file.read_range(b1, e1, &_array_range[b1]);
    for (difference_type i = e1; i < e0; i++) {
      _array_range[i] = _default;
    }
  }
  void flush() {
    if (_range_begin < _range_end) {
      _array_file.write_range(_range_begin, _range_end, &_array_range[_range_begin], _default);
    }
  }

private:
  Cache           _array_range;
  Data            _array_file;
  difference_type _range_size;
  difference_type _margin;
  value_type      _default;
  difference_type _range_begin;
  difference_type _range_end;

  void setup_array_range() {
    // range may be negative, so variables should be signed integers.
    _array_range.set_range((_range_begin-_margin), (_range_size+2*_margin), _default);
  }
};
}
#endif
