/*
 * array_range.hpp
 */
#ifndef ARRAY_RANGE_HPP
#define ARRAY_RANGE_HPP

namespace RFOLD {

template <typename T>
class ArrayRange {
public:
  //type definitions
  typedef Array<T>       Data;
  typedef T              value_type;
  typedef T*             iterator;
  typedef const T*       const_iterator;
  typedef T&             reference;
  typedef const T&       const_reference;
  typedef std::size_t    size_type;
  typedef std::ptrdiff_t difference_type;

  ArrayRange() 
    : _offset(0) {}
  ArrayRange(difference_type offset, size_type size)
    : _data(size), _offset(offset) {}
  ArrayRange(difference_type offset, size_type size, const T& t)
    : _data(size), _offset(offset) {fill(t);}
  ArrayRange(const ArrayRange& other) 
    : _data(other._data), _offset(other._offset) {
  }
  ArrayRange& operator=(const ArrayRange& other) {
    _data = other._data;
    _offset = other._offset;
    return *this;
  }
  virtual ~ArrayRange() {}
  bool operator==(const ArrayRange& other) const {
    if (_offset != other._offset) return false;
    if (size() != other.size()) return false;

    return std::equal(begin(), end(), other.begin());
  }
  bool operator!=(const ArrayRange& other) const {
    return !(*this == other);
  }

  //iterator support
  iterator begin() {return _data.begin();}
  const_iterator begin() const {return _data.begin();}
  iterator end() {return _data.end();}
  const_iterator end() const {return _data.end();}

  reference front() {return get(0);}
  const_reference front() const {return get(0);}
  reference back() {return get(size() - 1);}
  const_reference back() const {return get(size() - 1);}
  
  //direct element access
  reference operator[](difference_type i) {return get(i);}
  const_reference operator[](difference_type i) const {return get(i);}

  //size is constant
  difference_type size() const {return _data.size();}
  //size_type max_size() const {return _size;}

  //conversion to ordinary array
  T* data() {return _data.data();}

  bool empty() const {return _data.empty();}
  void clear() {
    _data.clear();
    _offset = 0;
  }
  void fill(const T& t) {std::fill(begin(), end(), t);}
  void swap(ArrayRange& other) {
    std::swap(_data, other._data);
    std::swap(_offset, other._offset);
  }
  void resize(difference_type size) {_data.resize(size);}
  void set_offset(difference_type offset) {_offset = offset;}
  Array<T> to_a() const {return _data;}
  bool includes_index(difference_type i) const {
    return (_offset <= i && i < (_offset + size()));}
  void get_range(difference_type& b, difference_type& e) const {
    b = _offset, e = (_offset + size());}
  difference_type index_begin() const {return _offset;}
  difference_type index_end() const {return (_offset + (size()));}
  void set_range(difference_type b, difference_type e, const T& val = T()) {
    _data.resize(e-b);
    _data.fill(val);
    _offset = b;
  }
  void copy_overlap(ArrayRange& other) {
    difference_type b = std::max(_offset, other._offset);
    difference_type e = std::min(_offset+size(), other._offset+other.size());
    for (difference_type i = b; i <= e; i++) {
      (*this)[i] = other[i];
    }
  }
  void shift_offset(difference_type offset, const T& val = T()) {
    if (offset < _offset) {
      difference_type n = (_offset - offset);
      if (n < _data.size) {
	for (difference_type i = (size()-1); i >= n; i--) {
	  _data[i] = _data[i-n];
	}
	for (difference_type i = (n-1); i >= 0; i--) {
	  _data[i] = val;
	}
      } else {
        _data.fill(val);
      }
    } else {
      difference_type n = (offset - _offset);
      if (n < _data.size) {
	for (difference_type i = 0; i <= (size()-n-1); i++) {
	  _data[i] = _data[i+n];
	}
	for (difference_type i = (size()-n); i <= (size()-1); i++) {
	  _data[i] = val;
	}
      } else {
        _data.fill(val);
      }
    }
    _offset = offset;
  }

protected:
  Data _data;
  difference_type _offset;

  inline reference get(difference_type i) {
    Assert(_offset <= i && i < (_offset + size()));
    return _data[i - _offset];
  }
  inline const_reference get(difference_type i) const {
    Assert(_offset <= i && i < (_offset + size()));
    return _data[i - _offset];
  }
};
}
#endif
