/*
 * tri_matrix_ring.hpp
 */
#ifndef TRI_MATRIX_RING_HPP
#define TRI_MATRIX_RING_HPP

namespace RFOLD {
template <typename T, typename Data = Array<T> >
class TriMatrixRing0 {
public:
  typedef typename Data::value_type value_type;
  typedef typename Data::iterator iterator;
  typedef typename Data::const_iterator const_iterator;
  typedef typename Data::reference reference;
  typedef typename Data::const_reference const_reference;
  typedef typename Data::size_type size_type;
  typedef typename Data::difference_type difference_type;

  TriMatrixRing0() : _nlayer(0), _length(0), _period(0), _stride(0), _area(0), _offset(0) {}
  TriMatrixRing0(const TriMatrixRing0& m) 
    : _data(m._data), _nlayer(m._nlayer), _length(m._length), _period(m._period), 
      _stride(m._stride), _area(m._area), _offset(m._offset) {}
  ~TriMatrixRing0() {}
  bool empty() const {return _data.empty();}
  int size() const {return _data.size();}
  int nlayer() const {return _nlayer;}
  int length() const {return _length;}
  int period() const {return _period;}
  int area() const {return _area;}
  void set_size(int nlayer, int length, int period) {
    Check(nlayer >= 0 && length >= 0 && period >= 0);
    _nlayer = nlayer;
    _length = length;
    _period = period;
    _stride = (2 * _length - 1);
    _area   = (_stride * _period);
    _offset = (_length - 1);
    _data.resize(_nlayer * _area);
  }
  const_reference get(int s, int i, int j, bool row_major = true) const {
    return _data[get_index(s, i, j, row_major)];}
  void set(int s, int i, int j, const_reference val) {
    _data[get_index(s, i, j, true)] = val;
    _data[get_index(s, i, j, false)] = val;}
  const_iterator iter(int s, int i, int j, bool row_major = true) const {
    return (_data.begin() + get_index(s, i, j, row_major));}
  void fill(const_reference val) {_data.fill(val);}
  void clear() {
    _data.clear(); _nlayer = 0; _length = 0; 
    _period = 0; _stride = 0; _area = 0, _offset = 0;}
  int get_index(int s, int i, int j, bool row_major = true) const {
#ifdef DEBUG
    if (!((0 <= s && s < _nlayer) && (i <= j && (j - i) < _length))) {
      Die("[%d,%d,%d,%d] out of range [%d,%d]", s, i, j, row_major, _nlayer, _length);
    }
#endif
    if (row_major) {
      return (s * _area + _offset + _stride * (i % _period) + (j - i));
    } else {
      return (s * _area + _offset + _stride * (j % _period) - (j - i));
    }
  }
  void add(int s, int i, int j, const_reference val) {
    _data[get_index(s, i, j, true)] += val;
    _data[get_index(s, i, j, false)] += val;}
  void logadd(int s, int i, int j, const_reference val) {
    LOGADD(_data[get_index(s, i, j, true)], val);
    LOGADD(_data[get_index(s, i, j, false)], val);
  }
  string to_s() const {return "";}
  void print() const {cout << to_s() << flush;}
private:
  Data _data;
  int _nlayer;
  int _length;
  int _period;
  int _stride;
  int _area;
  int _offset;
};
}
namespace RFOLD{
template <typename T, typename Data = Array<T> >
class TriMatrixRing1 {
public:
  typedef typename Data::value_type value_type;
  typedef typename Data::iterator iterator;
  typedef typename Data::const_iterator const_iterator;
  typedef typename Data::reference reference;
  typedef typename Data::const_reference const_reference;
  typedef typename Data::size_type size_type;
  typedef typename Data::difference_type difference_type;

  TriMatrixRing1() 
    : _nlayer(0), _length(0), _period(0), _area(0), _row_major(true) {}
  TriMatrixRing1(const TriMatrixRing1& m) 
    : _data(m._data), 
      _nlayer(m._nlayer), _length(m._length), _period(m._period), 
      _area(m._area), _row_major(m._row_major) {}
  ~TriMatrixRing1() {}
  bool empty() const {return _data.empty();}
  int size() const {return _data.size();}
  int nlayer() const {return _nlayer;}
  int length() const {return _length;}
  int period() const {return _period;}
  int area() const {return _area;}
  bool is_row_major() const {return _row_major;}
  void set_size(int nlayer, int length, int period, bool row_major) {
    Check(nlayer >= 0 && length >= 0 && period >= 0);
    _nlayer = nlayer;
    _length = length;
    _period = period;
    _row_major = row_major;
    _area   = (_length * _period);
    _data.resize(_nlayer * _area);
  }
  const_reference get(int s, int i, int j) const {
    return _data[get_index(s, i, j, _row_major)];}
  void set(int s, int i, int j, const_reference val) {
    _data[get_index(s, i, j, _row_major)] = val;}
  const_iterator iter(int s, int i, int j) const {
    return (_data.begin() + get_index(s, i, j, _row_major));}
  void fill(const_reference val) {_data.fill(val);}
  void clear() {
    _data.clear(); 
    _nlayer = 0; _length = 0; _period = 0; _area = 0; _row_major = true;}
  int get_index(int s, int i, int j, bool row_major) const {
#ifdef DEBUG
    if (!((0 <= s && s < _nlayer) && (i <= j && (j - i) < _length))) {
      Die("[%d,%d,%d,%d] out of range [%d,%d]", s, i, j, _row_major, _nlayer, _length);
    }
#endif
    if (row_major) {
      return (s * _area + (0) + _length * (i % _period) + (j - i));
    } else {
      return (s * _area + (_length - 1) + _length * (j % _period) - (j - i));
    }
  }
  void add(int s, int i, int j, const_reference val) {
    _data[get_index(s, i, j, _row_major)] += val;}
  void logadd(int s, int i, int j, const_reference val) {
    LOGADD(_data[get_index(s, i, j, _row_major)], val);}
  void flip_order_forward(int i0) {
    if (_row_major) {
      flip_order_row_major(i0 + _period - 1);
    } else {
      flip_order_col_major(i0);
    }
  }
  void flip_order_backward(int j0) {
    if (_row_major) {
      flip_order_row_major(j0);
    } else {
      flip_order_col_major(j0 - _period + 1);
    }
  }
  void flip_order_row_major(int j0) {
    for (int s = 0; s < _nlayer; s++) {
      for (int j = j0; j > (j0 - _period); j--) {
	for (int i = j; i > max((j - _length), (j0 - _period)); i--) {
	  _data[get_index(s, i, j, false)] = _data[get_index(s, i, j, true)];
	}
      }
    }
    _row_major = !_row_major;
  }
  void flip_order_col_major(int i0) {
    for (int s = 0; s < _nlayer; s++) {
      for (int i = i0; i < (i0 + _period); i++) {
	for (int j = i; j < min((i + _length), (i0 + _period)); j++) {
	  _data[get_index(s, i, j, true)] = _data[get_index(s, i, j, false)];
	}
      }
    }
    _row_major = !_row_major;
  }
  string to_s() const {return "";}
  void print() const {cout << to_s() << flush;}
private:
  Data _data;
  int _nlayer;
  int _length;
  int _period;
  int _area;
  bool _row_major;
};
}
#endif
