/*
 * ct_file.hpp
 */
#ifndef CT_FILE_HPP
#define CT_FILE_HPP
#include "./array_file.hpp"

namespace RFOLD{
class CTFile {// 1 based indexing
public:
  typedef int T;
  typedef T              value_type;
  typedef T*             iterator;
  typedef const T*       const_iterator;
  typedef T&             reference;
  typedef const T&       const_reference;
  typedef std::size_t    size_type;
  typedef std::ptrdiff_t difference_type;
  enum {
    INNER_LOOP = (-1),
    OUTER_LOOP = 0 // 1 based is important. do not pairs with index = 0
  };

  CTFile() {
    _data.set_range_size(1000);
    _data.set_margin(10);
    _data.set_default(OUTER_LOOP);
  }
  CTFile(const std::string& filename) : _data(filename, 1000, 10, OUTER_LOOP) {}
  bool empty() const {return ((int)_data.size() <= 1);}   // OUTER_LOOP is padded at index = 0
  T size() const {return max(0, ((int)_data.size() - 1));}// OUTER_LOOP is padded at index = 0
  void resize(T n) {_data.resize(n + 1);}    // OUTER_LOOP is padded at index = 0
  void clear() {_data.clear();}
  void reset(const std::string& filename, T range_size, T margin) {
    _data.reset(filename, range_size, margin, OUTER_LOOP);}
  void set_file(const std::string& filename) {_data.set_file(filename);}
  void set_margin(T margin) {_data.set_margin(margin);}
  void set_range_size(T range_size) {_data.set_range_size(range_size);}
  void set_default() {_data.fill(OUTER_LOOP);}
  T range_begin() const {return _data.range_begin();}
  T range_end() const {return _data.range_end();}
  void load_range(T b, T e) {Assert(1 <= b); _data.load_range(b, e); return;}
  void load_range_if_needed(T b, T e) {Assert(1 <= b); _data.load_range_if_needed(b, e); return;}
  reference operator[](T i) {return _data[i];}
  const_reference operator[](T i) const {return _data[i];}
  bool pairs(T b, T e) const {return (_data[b] == e);}
  bool is_outer_loop(T i) const {return (_data[i] == OUTER_LOOP);}
  bool is_inner_loop(T i) const {return (_data[i] == INNER_LOOP);}
  bool is_loop(T i) const {return !is_stem(i);}
  bool is_stem(T i) const {return (_data[i] > 0);}// 1 based is important.
  bool is_left_stem(T i) const {return (0 < i && i < _data[i]);}
  bool is_right_stem(T i) const {return (0 < _data[i] && _data[i] < i);}
  int pair_pos(T i) const {Check(is_stem(i)); return _data[i];}
  void add_pair(T b, T e) {
    _data.load_range_if_needed(b, (e+1));
    if (is_outer_loop(b) && is_outer_loop(e)) {
      _data[b] = e;
      _data[e] = b;
      for (T i = (b+1); i <= (e-1); i++) {
	Assert(is_left_stem(i) || is_outer_loop(i));
	if (is_left_stem(i)) {
	  i = _data[i];
	} else {
	  _data[i] = INNER_LOOP;
	}
      }
    } else if (is_inner_loop(b) && is_inner_loop(e)) {
      _data[b] = e;
      _data[e] = b;
    } else {
      Die("cannot add pair at [%d,%d] [%d, %d]", b, e, _data[b], _data[e]);
    }
  }
  void unpair(T i) {
    _data.load_range_if_needed(i, i+1);
    if (!is_stem(i)) return;

    T b = std::min(i, _data[i]);
    T e = std::max(i, _data[i]);
    bool is_outer = true;
    for (T i = (b-1); i >= 1; i--) {
      _data.load_range_if_needed(i, i+1);
      if (is_outer_loop(i)) {
	break;
      } else if (is_inner_loop(i) || is_left_stem(i)) {
	is_outer = false;
	break;
      } else if (is_right_stem(i)) {
	i = _data[i];
      } else {
	Die("bad ct type ct[%d] = %d", i, _data[i]);
      }
    }
    if (is_outer) {
      _data.load_range_if_needed(b, b+1);
      _data[b] = OUTER_LOOP;
      _data.load_range_if_needed(e, e+1);
      _data[e] = OUTER_LOOP;
      for (T i = (b+1); i <= (e-1); i++) {
	_data.load_range_if_needed(i, i+1);
	Assert(is_left_stem(i) || is_inner_loop(i));
	if (is_left_stem(i)) {
	  i = _data[i];
	} else {
	  _data[i] = OUTER_LOOP;
	}
      }
    } else {
      _data.load_range_if_needed(b, b+1);
      _data[b] = INNER_LOOP;
      _data.load_range_if_needed(e, e+1);
      _data[e] = INNER_LOOP;
    }
  }
  void remove_distant_pairs(T max_pair_dist) {
    for (T i = 1; i <= (T)size(); i++) {
      _data.load_range_if_needed(i, i+1);
      if (is_left_stem(i)) {
	T dist = (_data[i] - i);
	if (max_pair_dist < dist) {
	  unpair(i);
	}
      }
    }
  }
  void remove_close_pairs(T min_pair_dist) {
    for (T i = 1; i <= (T)size(); i++) {
      _data.load_range_if_needed(i, i+1);
      if (is_left_stem(i)) {
	T dist = (_data[i] - i);
	if (dist < min_pair_dist) {
	  unpair(i);
	}
      }
    }
  }
  void write_local_structures(std::ostream& fo) {
    for (T i = 1; i <= (T)size(); i++) {
      _data.load_range_if_needed(i, i+1);
      Assert(is_left_stem(i) || is_outer_loop(i));
      if (is_outer_loop(i)) continue;

      T b = i;
      T e = _data[i];
      fo << setw(4) << b << " " << setw(4) << e << " ";
      for (T j = b; j <= e; j++) {
	_data.load_range_if_needed(j, j+1);
	fo << get_sscons_char(j);
      }
      fo << "\n";
      i = e;
    }
  }
  void write_sscons(std::ostream& fo, int width = 0) {
    for (T i = 1; i <= (T)size(); i++) {
      _data.load_range_if_needed(i, i+1);
      fo << get_sscons_char(i);
      if (width > 0 && (i % width == 0)) {
	fo << '\n';
      }
    }
    if (width > 0) {
      fo << '\n';
    }
    fo << std::flush;
  }
  char get_sscons_char(T i) const {
    char left_stem_char = '<';
    char right_stem_char = '>';
    char loop_char = '.';
    if (is_left_stem(i)) {
      return left_stem_char;
    } else if (is_right_stem(i)) {
      return right_stem_char;
    } else {
      return loop_char;
    }
  }
  std::string sscons() {std::ostringstream oss; write_sscons(oss); return oss.str();}
  std::string to_s() {return sscons();}
  void print() {write_sscons(std::cout);}
  void read_structure(std::istream& fi)  {
    char c = 0;
    std::vector<int> arr;
    int length = size(); // sequence length
    set_default();
    for (int i = 1; i <= length; i++) {// 1 based index
      if (!fi.get(c)) {
	Die("structure size %d < sequence size %d", (i-1), length);
      }
      switch (c) {
      case '<': case '(': case '[': // left char
	// i is left stem
	arr.push_back(i);
        break;
      case '>': case ')': case ']': // right char
	Check(!arr.empty(), "closing bracket %c at position %d has no pair", c, (i+1));
	add_pair(arr.back(), i);
	arr.pop_back();
	break;
      case ' ': case '\f': case '\n': case '\r': case '\t': case '\v'://space chars
	(--i);
	break;
      default:
	// inner and outer loop cases
	break;
      }
    }
    if (!arr.empty()) {
      Die("opening bracket at position %d has no pair", (arr.back() + 1));
    }
  }
  template <typename InpIt>
  void assign(InpIt first, InpIt last) {}

private:
  ArrayFile<T> _data;
};
}
#endif
