/*
 * energy_model.hpp
 *
 */
#ifndef ENERGY_MODEL
#define ENERGY_MODEL

#include "util.hpp"
#include "alpha.hpp"
#include "tensor.hpp"

namespace RFOLD {
class EnergyModel {
public:
  // typedef double ValueT;
  typedef ScoreT ValueT; // needs definition of ScoreT

  static ValueT GAS_CONST_KCAL_MOL_K() {return (1.9872 * 0.001);}
  static ValueT TEMPERATURE_K() {return (273.15 + 37.0);}
  static ValueT RT_KCAL_MOL() {return GAS_CONST_KCAL_MOL_K() * TEMPERATURE_K();}
  static ValueT energy_to_score(ValueT energy) {
    return (-energy / RT_KCAL_MOL());
  }
  static ValueT score_to_energy(ValueT score) {
    return (-score * RT_KCAL_MOL());
  }
  enum {
    C = 30,
    D = 0,
    M = Alpha::NNCODE,
    MIN_NSTACK = 0, //0 <= MIN_NSTACK <= D should be satisfied
    MIN_PAIR_DIST = 3,
    D0 = 0
  };
  enum State {
    STEM       ,
    STEM_END   ,
    MULTI      ,
    MULTI1     ,
    MULTI2     ,
    MULTI_BF   ,
    INNER_BEG  ,
    OUTER      ,

    HAIRPIN_END,
    OUTER_BEG  ,
    OUTER_END  ,

    STATE_X = -1
  };
  enum {
    NSTATE  = (INNER_BEG - STEM + 1),
    NSTATE1 = (OUTER - STEM + 1),
    NSTATE0 = (OUTER_END - STEM + 1)
  };
  typedef CArray<string, NSTATE0> StateToStrs;
  static const StateToStrs STATE_TO_STRS;
  static string state_to_s(int s) {return STATE_TO_STRS[s];}

  enum Dst {
    DST_C,
    DST_N,
    DST_S,
    DST_E,
    DST_W,
    DST_NE,
    DST_NW,
    DST_SE,
    DST_SW,
    DST_X = (-1)
  };
  enum {
    NDST = (DST_SW - DST_C + 1)
  };

  // S=STEM, E=STEM_END, H=Hairpin, I=Interior loop
  // M=MULTI, M1=MULTI1 O=OUTER, X=OUTER_BEG, OUTER_END, 
  // BF, BFL, BFR = bifurcation, L=left, R=right
  enum TrType { 
    TR_S_S  ,
    TR_S_E  ,
 
    TR_M_BF ,
    TR_M_BFL,
    TR_M_BFR,
  
    TR_M2_S ,
    TR_M2_M2,

    TR_M1_M2,
    TR_M1_MBF,

    TR_M_M   ,
    TR_M_MBF ,

    TR_E_H  ,
    TR_E_I  ,
    TR_E_M  ,

    TR_IB_S ,

    TR_O_X  ,
    TR_X_O  ,
    TR_O_O  ,
    TR_O_IB ,
    TR_O_BF ,
    TR_O_BFL,
    TR_O_BFR
  };
  enum {
    NTRANSITION = (TR_O_BFR - TR_S_S + 1)
  };
  enum BfType {
    BF_NONE,
    BF_PARENT,
    BF_LEFT,
    BF_RIGHT
  };
  struct Transition {
    int type;
    BfType bf;
    int from;
    int to;
    int to1;
    int dst_in;
    int dst_out;
    string name;
  };
  typedef CArray<Transition, NTRANSITION> Transitions;
  static const Transitions TRANSITIONS;
  class TBNode;

  // template <bool fwd> then MULTI1 <=> MULTI2
  static int inside_layer(int s) {
    switch (s) {
    case STEM:     return 0;
    case STEM_END: return 1;
    case MULTI:    return 2;
    case MULTI1:   return 3;
    case MULTI2:   return 4;
    case INNER_BEG: return 5;
    default:       return (-1);
    }
  }
  static int outside_layer(int s) {
    switch (s) {
    case STEM_END: return 0;
    case MULTI_BF: return 1;
    default: return (-1);
    }
  }
  // template <bool fwd> then MULTI1 <=> MULTI2
  static int cyk_layer(int s) {
    switch (s) {
    case STEM:   return 0;
    case MULTI1: return 1;
    case MULTI2: return 2;
    default: return (-1);
    }
  }
  enum {
    NLAYER_INSIDE  = 6,
    NLAYER_OUTSIDE = 2,
    NLAYER_CYK     = 3
  };

  // nussinov state transitions
  enum NusState {
    NUS_PAIR,
    NUS_INNER,
    NUS_OUTER,
    NUS_INNER_END,
    NUS_OUTER_END,
    NUS_STATE_X = -1
  };
  enum {
    NUS_NSTATE  = (NUS_INNER - NUS_PAIR + 1),
    NUS_NSTATE1 = (NUS_OUTER - NUS_PAIR + 1),
    NUS_NSTATE0 = (NUS_OUTER_END - NUS_PAIR + 1)
  };
  enum NusTrType {
     NUS_TR_X,
     NUS_TR_P,
     NUS_TR_L,
     NUS_TR_R,
     NUS_TR_B,
     NUS_TR_I,
     NUS_TR_O_X,
     NUS_TR_O_O,
     NUS_TR_O_P,
     NUS_TR_O_B
  };
  enum {
    NUS_NTRANSITION = (NUS_TR_O_B - NUS_TR_X + 1)
  };
  typedef CArray<Transition, NUS_NTRANSITION> NusTransitions;
  static const NusTransitions NUS_TRANSITIONS;
  class NusTBNode;
  static int nus_cyk_layer(int s) {return (s == NUS_INNER ? 0 : (-1));}
  enum {NUS_NLAYER_CYK = 1};

  enum ParamGroupType {
    large_loop_factor,
    multiloop_offset,
    multiloop_nuc,
    multiloop_helix,
    hairpinloop_change,
    bulgeloop_change,
    internalloop_change,
    internalloop_asymmetry,
    terminal_base_pair,
    dangle3,
    dangle5,
    stack, 
    tstackh,
    tstacki,
    tstackm,
    // tstacke, 
    int2, 
    int21,
    int22,
    triloop,
    tetraloop,

    outer_unpaired,
    outer_branch,
    // outer_branch_length_factor,
#ifdef USE_TSC_FREQ
    nuc_freq,
    dinuc_freq,
#endif
    outer_nuc_freq,
    outer_dinuc_freq,

    hairpinloop_cumulative_change,
    bulgeloop_cumulative_change,
    internalloop_cumulative_change,
    internalloop_cumulative_asymmetry,
    param_group_null = (-1)
  };
  enum {
    // NPARAMGROUP = (tetraloop - large_loop_factor + 1),
    // NPARAMGROUP = (dinuc_freq - large_loop_factor + 1),
    NPARAMGROUP = (outer_dinuc_freq - large_loop_factor + 1),
    NPARAMGROUP1 = (internalloop_cumulative_asymmetry - large_loop_factor + 1)
  };
  enum SubscriptType {
    ST_NONE,
    ST_ALPHA,
    ST_LENGTH
  };
  enum AlphaType {
    AT_NONE,
    AT_NCODE,
    AT_PCODE,
    AT_SCODE
  };
  struct ParamGroup {
    enum {MAX_RANK = 4};
    typedef CArray<int, MAX_RANK+1> Lengths;
    typedef CArray<AlphaType, MAX_RANK+1> AlphaTypes;
    ParamGroupType type;
    string name;
    ParamGroupType pair_group;
    Lengths lengths; // fill (-1) for rank <= i
    SubscriptType subscript_type;
    AlphaTypes alpha_types;
    bool symmetric;
    int base_subscript;
    int sum_bound;
    ValueT default_value;
    int rank() const {
      for (int k = 0; k < (int)lengths.size(); k++) {
	if (lengths[k] < 0) return k;
      }
      return lengths.size();
    }
  };
  typedef CArray<ParamGroup, NPARAMGROUP1> ParamGroups;
  static const ParamGroups PARAM_GROUPS;
  class FeatureCounter;
  class IndexToParam;
  class Freq {
  public:
    typedef CArray<ScoreT, Alpha::NPCODE0> P;
    typedef CArray<ScoreT, Alpha::NNCODE0> S;
    P p;
    S s;
    void clean() {
      p.fill(0);
      s.fill(0);
    }
    bool equal(const Freq& other) const {
      return (std::equal(p.begin(), p.end(), other.p.begin())
	      && std::equal(s.begin(), s.end(), other.s.begin()));
    }
    string to_s() const {
      ostringstream oss;
      oss << "p: \n";
      for (int i = 0; i < (int)p.size(); i++) {
	oss << "  " << Alpha::pcode_to_str(i) << ": " << p[i] << "\n";
      }
      oss << "s: \n";
      for (int i = 0; i < (int)s.size(); i++) {
	oss << "  " << Alpha::ncode_to_str(i) << ": " << s[i] << "\n";
      }
      return oss.str();
    }
  };
  class CurrCode {
  public:
    typedef Alpha::CodeT N;
    typedef Alpha::CodeT P;
    typedef Alpha::CodeT S;
    N ni;
    N nj;
    P p;
    S s;
    void clean() {
      ni = nj = Alpha::N_BAD;
      p = Alpha::P_BAD;
      s = Alpha::S_BAD;
    }
  };
};
}

namespace RFOLD {
const EnergyModel::StateToStrs
EnergyModel::
STATE_TO_STRS = {{
  "STEM", 
  "STEM_END",
  "MULTI",
  "MULTI1",
  "MULTTI2",
  "MULTI_BF"   ,
  "INNER_BEG"  ,
  "OUTER",
  "HAIRPIN_END",
  "OUTER_BEG",
  "OUTER_END"
}};

const EnergyModel::Transitions
EnergyModel::
TRANSITIONS = {{
#define Stem(n) ((State)(STEM + (n < D ? n : D)))
#define Item(type, bf, from, to, to1, dst_in, dst_out) \
  {type, bf, from, to, to1,dst_in,dst_out, string(#type)}
  Item(TR_S_S   ,  BF_NONE,   STEM ,     Stem(1),     STATE_X, DST_S, DST_N),
  Item(TR_S_E   ,  BF_NONE,   STEM,      STEM_END,    STATE_X, DST_S, DST_N),
	        	  
  Item(TR_M_BF  ,  BF_PARENT, MULTI_BF,  MULTI1,     MULTI2,     DST_X, DST_X),    
  Item(TR_M_BFL ,  BF_LEFT,   MULTI_BF,  MULTI1,     MULTI2,     DST_X, DST_X),    
  Item(TR_M_BFR ,  BF_RIGHT,  MULTI_BF,  MULTI1,     MULTI2,     DST_X, DST_X),    
	        	  			     	                                
  Item(TR_M2_S  ,  BF_NONE,   MULTI2,    STEM,       STATE_X, DST_C, DST_C),    
  Item(TR_M2_M2 ,  BF_NONE,   MULTI2,    MULTI2,     STATE_X, DST_SW, DST_NE),  
	        	  			     	                                
  Item(TR_M1_M2 ,  BF_NONE,   MULTI1,    MULTI2,     STATE_X, DST_C, DST_C),    
  Item(TR_M1_MBF,  BF_NONE,   MULTI1,    MULTI_BF,   STATE_X, DST_C, DST_C),    
		  		       		     	                                
  Item(TR_M_M   ,  BF_NONE,   MULTI,     MULTI,      STATE_X, DST_SE, DST_NW),  
  Item(TR_M_MBF ,  BF_NONE,   MULTI,     MULTI_BF,   STATE_X, DST_C, DST_C),    
	        	          	  	     	                                
  Item(TR_E_H   ,  BF_NONE,   STEM_END,  HAIRPIN_END,STATE_X, DST_C, DST_C),    
  Item(TR_E_I   ,  BF_NONE,   STEM_END,  STEM,       STATE_X, DST_X, DST_X),    
  Item(TR_E_M   ,  BF_NONE,   STEM_END,  MULTI,      STATE_X, DST_C, DST_C),    
	        	  		       	     	                                   
  Item(TR_IB_S  ,  BF_NONE,   INNER_BEG, STEM,       STATE_X, DST_C, DST_C),    
	        	  		       	     	                                
  Item(TR_O_X   ,  BF_NONE,   OUTER,     OUTER_END,  STATE_X, DST_C, DST_C),    
  Item(TR_X_O   ,  BF_NONE,   OUTER_BEG, OUTER,      STATE_X, DST_C, DST_C),    
  Item(TR_O_O   ,  BF_NONE,   OUTER,     OUTER,      STATE_X, DST_X, DST_X),    
  Item(TR_O_IB  ,  BF_NONE,   OUTER,     INNER_BEG,  STATE_X, DST_C, DST_C),    
						     	                                
  Item(TR_O_BF  ,  BF_PARENT, OUTER,     OUTER,      INNER_BEG,  DST_X, DST_X),    
  Item(TR_O_BFL ,  BF_LEFT,   OUTER,     OUTER,      INNER_BEG,  DST_X, DST_X),    
  Item(TR_O_BFR ,  BF_RIGHT,  OUTER,     OUTER,      INNER_BEG,  DST_X, DST_X)     
#undef Item
#undef Stem
}};

class 
EnergyModel::
TBNode {
public:
  enum Type {
    S_S = 0,
    S_E = 1,
    M_M = 0,
    M_S = 1,
    M_B = 1,
    E_H = 0,
    E_I = 1,
    E_M = 2
  };
  static const TBNode& null() {
    static const TBNode nd = {S_S};
    return nd;
  }
  void clean() {(*this) = null();}
  void set(TrType type, int i, int j, int k, int l) {
    switch (type) {
    case TR_S_S  : _stem = S_S; break;
    case TR_S_E  : _stem = S_E; break;
    case TR_M_BF : _multi_bf = k; break;
    case TR_M2_S : _multi2 = M_S; break;
    case TR_M2_M2: _multi2 = M_M; break;
    case TR_M1_M2: _multi1 = M_M; break;
    case TR_M1_MBF: _multi1 = M_B; break;
    case TR_M_M   : _multi = M_M; break;
    case TR_M_MBF : _multi = M_B; break;
    case TR_E_H  : _stem_end = E_H; break;
    case TR_E_I  : 
      _stem_end = E_I; 
      _internal_li = abs(k-i);
      _internal_lj = abs(l-j); 
      Assert(C < 256 && _internal_li <= C && _internal_lj <= C);
      break;
    case TR_E_M  : _stem_end = E_M; break;
    case TR_IB_S : break;
    default:
      Die("bad transition type %d", type);
    }
  }
  std::pair<int, int> pos(State s, int i, int j) const {
    switch (s) {
    case STEM     : return make_pair(i+1, j-1);
    case STEM_END :
      switch (_stem_end) {
      case E_H: return make_pair(i, j);
      case E_I: return make_pair(i+_internal_li, j-_internal_lj);
      case E_M: return make_pair(i, j);
      default: 
	Die("bad stem_end %d", _stem_end); 
	return make_pair(-1, -1);
      }
    case MULTI    : return (_multi == M_M ? make_pair(i+1, j) : make_pair(i, j));
    case MULTI1   : return make_pair(i, j);
    case MULTI2   : return (_multi2 == M_M ? make_pair(i, j-1) : make_pair(i, j));
    case MULTI_BF : return make_pair(_multi_bf, _multi_bf);
    case INNER_BEG: return make_pair(i, j);
    default: return make_pair((-1), (-1));
    }
  }
  const Transition& transition(State s) const {
    TrType type = TR_O_IB;
    switch (s) {
    case STEM     : type = (_stem  == S_S ? TR_S_S  : TR_S_E ); break;
    case STEM_END :
      switch (_stem_end) {
      case E_H: type = TR_E_H; break;
      case E_I: type = TR_E_I; break;
      case E_M: type = TR_E_M; break;
      default: Die("bad stem_end %d", _stem_end); break;
      }
      break;
    case MULTI    : type = (_multi == M_M ? TR_M_M : TR_M_MBF); break;
    case MULTI1   : type = (_multi1 == M_M ? TR_M1_M2 : TR_M1_MBF); break;
    case MULTI2   : type = (_multi2 == M_M ? TR_M2_M2 : TR_M2_S); break;
    case MULTI_BF : type = TR_M_BF; break;
    case INNER_BEG: type = TR_IB_S; break;
    default: Die("bad state %d", s); break;
    }
    return TRANSITIONS[type];
  }
  unsigned int  _multi_bf;
  unsigned char _internal_li;
  unsigned char _internal_lj;
  unsigned int  _stem_end :2;
  unsigned int  _stem     :1;
  unsigned int  _multi    :1;
  unsigned int  _multi1   :1;
  unsigned int  _multi2   :1;
};
const EnergyModel::NusTransitions
EnergyModel::
NUS_TRANSITIONS = {{
#define Item(type, bf, from, to, to1, dst_in, dst_out) \
  {type, bf, from, to, to1,dst_in,dst_out, string(#type)}
  Item(NUS_TR_X  , BF_NONE,   NUS_INNER, NUS_INNER_END, NUS_STATE_X, DST_C, DST_C),   
  Item(NUS_TR_P  , BF_NONE,   NUS_PAIR,  NUS_INNER,     NUS_STATE_X, DST_S, DST_N),   
  Item(NUS_TR_L  , BF_NONE,   NUS_INNER, NUS_INNER,     NUS_STATE_X, DST_SE, DST_NW), 
  Item(NUS_TR_R  , BF_NONE,   NUS_INNER, NUS_INNER,     NUS_STATE_X, DST_SW, DST_NE), 
  Item(NUS_TR_B  , BF_PARENT, NUS_INNER, NUS_INNER,     NUS_INNER,   DST_X, DST_X),   
  Item(NUS_TR_I  , BF_NONE,   NUS_INNER, NUS_PAIR,      NUS_STATE_X, DST_C, DST_C),   
  Item(NUS_TR_O_X, BF_NONE,   NUS_OUTER, NUS_OUTER_END, NUS_STATE_X, DST_C, DST_C),   
  Item(NUS_TR_O_O, BF_NONE,   NUS_OUTER, NUS_OUTER,     NUS_STATE_X, DST_X, DST_X),   
  Item(NUS_TR_O_P, BF_NONE,   NUS_OUTER, NUS_PAIR,      NUS_STATE_X, DST_C, DST_C),   
  Item(NUS_TR_O_B, BF_PARENT, NUS_OUTER, NUS_OUTER,     NUS_PAIR,    DST_X, DST_X)     
#undef Item
}};
class 
EnergyModel::
NusTBNode {
public:
  enum Type {
    TB_P = (-1),
    TB_L = (-2),
    TB_R = (-3),
    TB_X = (-4)
  };
  static const NusTBNode& null() {
    static const NusTBNode nd = {TB_X};
    return nd;
  }
  void clean() {(*this) = null();}
  void set(NusTrType type, int i, int j, int k, int l) {
    switch (type) {
    case NUS_TR_X: _data = TB_X; return;
    case NUS_TR_P:               return;
    case NUS_TR_L: _data = TB_L; return;
    case NUS_TR_R: _data = TB_R; return;
    case NUS_TR_B: _data = k;    return;
    case NUS_TR_I: _data = TB_P; return;
    default:
      Die("bad transition type %d", type); return;
    }
  }
  std::pair<int, int> pos(NusState s, int i, int j) const {
    switch (s) {
    case NUS_PAIR: return make_pair(i+1, j-1);
    case NUS_INNER:
      switch (_data) {
      case TB_P: return make_pair(i, j);
      case TB_L: return make_pair(i+1, j);
      case TB_R: return make_pair(i, j-1);
      case TB_X: return make_pair(i, j);
      default: return make_pair(_data, _data);
      }	
    default: return make_pair(i, j);
    }
  }
  const Transition& transition(NusState s) const {
    NusTrType type = NUS_TR_X;
    switch (s) {
    case NUS_PAIR: type = NUS_TR_P; break;
    case NUS_INNER:
      switch (_data) {
      case TB_P: type = NUS_TR_I; break;
      case TB_L: type = NUS_TR_L; break;
      case TB_R: type = NUS_TR_R; break;
      case TB_X: type = NUS_TR_X; break;
      default: type = NUS_TR_B;   break;
      }	
    default: break;
    }
    return NUS_TRANSITIONS[type];
  }
  int _data;
};
const EnergyModel::ParamGroups
EnergyModel::
PARAM_GROUPS = {{
#define Pair(t) t, string(#t)
{Pair(large_loop_factor),                 param_group_null,{{(-1)}},
                                          ST_NONE,{{}},false,(-1),(-1),0.0},
{Pair(multiloop_offset),                  param_group_null,{{(-1)}},
                                          ST_NONE,{{}},false,(-1),(-1),0.0},
{Pair(multiloop_nuc),                     param_group_null,{{(-1)}},
                                          ST_NONE,{{}},false,(-1),(-1),0.0},
{Pair(multiloop_helix),                   param_group_null,{{(-1)}},
                                          ST_NONE,{{}},false,(-1),(-1),0.0},

{Pair(hairpinloop_change),                hairpinloop_cumulative_change,    {{(C+1),(-1)}},
                                          ST_LENGTH,{{}},false,(-1),(-1),0.0},
{Pair(bulgeloop_change),                  bulgeloop_cumulative_change,      {{(C+1),(-1)}},
                                          ST_LENGTH,{{}},false,(-1),(-1),0.0},
{Pair(internalloop_change),               internalloop_cumulative_change,   {{(C+1),(-1)}},
                                          ST_LENGTH,{{}},false,(-1),(-1),0.0},
{Pair(internalloop_asymmetry),            internalloop_cumulative_asymmetry,{{(C+1),(-1)}},
                                          ST_LENGTH,{{}},false,(-1),(-1),0.0},

{Pair(terminal_base_pair),        	  param_group_null,{{Alpha::NSCODEN,(-1)}},
                                          ST_ALPHA,
                                          {{AT_SCODE}},false,(-1),(-1),-9999.9},
{Pair(dangle3),           	          param_group_null,{{Alpha::NSCODEN, Alpha::NNCODEN,(-1)}},
                                          ST_ALPHA,
                                          {{AT_SCODE,AT_NCODE}},false,(-1),(-1),0.0},
{Pair(dangle5),          	          param_group_null,
                                          {{Alpha::NSCODEN, Alpha::NNCODEN,(-1)}},
                                          ST_ALPHA,
                                          {{AT_SCODE,AT_NCODE}},false,(-1),(-1),0.0},
{Pair(stack),                   	  param_group_null,
                                          {{Alpha::NSCODEN, Alpha::NSCODEN,(-1)}},
                                          ST_ALPHA,
                                          {{AT_SCODE,AT_SCODE}}, true,(-1),(-1),-9999.9},
#ifndef USE_TSTACK4
{Pair(tstackh),           	          param_group_null,
                                          {{Alpha::NSCODEN, Alpha::NPCODEN,(-1)}},
                                          ST_ALPHA,
                                          {{AT_SCODE,AT_PCODE}},false,(-1),(-1),0.0},
{Pair(tstacki),           	          param_group_null,
                                          {{Alpha::NSCODEN, Alpha::NPCODEN,(-1)}},
                                          ST_ALPHA,
                                          {{AT_SCODE,AT_PCODE}},false,(-1),(-1),0.0},
{Pair(tstackm),           	          param_group_null,
                                          {{Alpha::NSCODEN, Alpha::NPCODEN,(-1)}},
                                          ST_ALPHA,
                                          {{AT_SCODE,AT_PCODE}},false,(-1),(-1),0.0},
  // {Pair(tstacke),           	          param_group_null,
  //                                          {{Alpha::NSCODEN, Alpha::NPCODEN,(-1)}},
  //                                          ST_ALPHA,
  //                                          {{AT_SCODE,AT_PCODE}},false,(-1),(-1),0.0},
#else
{Pair(tstackh),           	          param_group_null,
                                          {{Alpha::NNCODEN, Alpha::NNCODEN, Alpha::NNCODEN, Alpha::NNCODEN, (-1)}},
                                          ST_ALPHA,
					  {{AT_NCODE,AT_NCODE, AT_NCODE, AT_NCODE}},false,(-1),(-1),0.0},
{Pair(tstacki),           	          param_group_null,
                                          {{Alpha::NNCODEN, Alpha::NNCODEN, Alpha::NNCODEN, Alpha::NNCODEN, (-1)}},
                                          ST_ALPHA,
                                          {{AT_NCODE,AT_NCODE, AT_NCODE, AT_NCODE}},false,(-1),(-1),0.0},
{Pair(tstackm),           	          param_group_null,
                                          {{Alpha::NNCODEN, Alpha::NNCODEN, Alpha::NNCODEN, Alpha::NNCODEN, (-1)}},
                                          ST_ALPHA,
                                          {{AT_NCODE,AT_NCODE, AT_NCODE, AT_NCODE}},false,(-1),(-1),0.0},
  //{Pair(tstacke),           	          param_group_null,
  //                                          {{Alpha::NNCODEN, Alpha::NNCODEN, Alpha::NNCODEN, Alpha::NNCODEN, (-1)}},
  //                                          ST_ALPHA,
  //                                          {{AT_NCODE,AT_NCODE, AT_NCODE, AT_NCODE}},false,(-1),(-1),0.0},
#endif

{Pair(int2),           	                  param_group_null,
                                          {{Alpha::NSCODEN, Alpha::NNCODEN, Alpha::NNCODEN, Alpha::NSCODEN,(-1)}},
                                          ST_ALPHA,
                                          {{AT_SCODE,AT_NCODE,AT_NCODE,AT_SCODE}},true,(-1),(-1),0.0},
{Pair(int21),           	          param_group_null,
                                          {{Alpha::NSCODEN, Alpha::NPCODEN, Alpha::NNCODEN, Alpha::NSCODEN,(-1)}},
                                          ST_ALPHA,
                                          {{AT_SCODE,AT_PCODE,AT_NCODE,AT_SCODE}},false,(-1),(-1),0.0},
{Pair(int22),           	          param_group_null,
                                          {{Alpha::NSCODEN, Alpha::NPCODEN, Alpha::NPCODEN, Alpha::NSCODEN,(-1)}},
                                          ST_ALPHA,
                                          {{AT_SCODE,AT_PCODE,AT_PCODE,AT_SCODE}},true,(-1),(-1),0.0},

{Pair(triloop),           	          param_group_null,
                                          {{Alpha::NSCODEN, Alpha::NPCODEN, Alpha::NNCODEN,(-1)}},
                                          ST_ALPHA,
                                          {{AT_SCODE,AT_PCODE,AT_NCODE}},false,(-1),(-1),0.0},
{Pair(tetraloop),           	          param_group_null,
                                          {{Alpha::NSCODEN, Alpha::NPCODEN, Alpha::NPCODEN,(-1)}},
                                          ST_ALPHA,
                                          {{AT_SCODE,AT_PCODE,AT_PCODE}},false,(-1),(-1),0.0},

{Pair(outer_unpaired),                    param_group_null,{{(-1)}},
                                          ST_NONE,{{}},false,(-1),(-1),0.0},
{Pair(outer_branch),                      param_group_null,{{(-1)}},
                                          ST_NONE,{{}},false,(-1),(-1),0.0},
// {Pair(outer_branch_length_factor),        param_group_null,{{(-1)}},
//                                           ST_NONE,{{}},false,(-1),(-1),0.0},
#ifdef USE_TSC_FREQ
{Pair(nuc_freq),                          param_group_null,
                                          {{Alpha::NNCODEN,(-1)}},
                                          ST_ALPHA,
                                          {{AT_NCODE}}, false, (-1),(-1),0.0},
{Pair(dinuc_freq),                        param_group_null,
                                          {{Alpha::NPCODEN,(-1)}},
                                          ST_ALPHA,
                                          {{AT_PCODE}}, false, (-1),(-1),0.0},
#endif
{Pair(outer_nuc_freq),                    param_group_null,
                                          {{Alpha::NNCODEN,(-1)}},
                                          ST_ALPHA,
                                          {{AT_NCODE}}, false, (-1),(-1),0.0},
{Pair(outer_dinuc_freq),                  param_group_null,
                                          {{Alpha::NPCODEN,(-1)}},
                                          ST_ALPHA,
                                          {{AT_PCODE}}, false, (-1),(-1),0.0},

{Pair(hairpinloop_cumulative_change),     hairpinloop_change,{{(C+1),(-1)}},
                                          ST_LENGTH,{{}},false,(-1),(-1),0.0},
{Pair(bulgeloop_cumulative_change),       bulgeloop_change,{{(C+1),(-1)}},
                                          ST_LENGTH,{{}},false,(-1),(-1),0.0},
{Pair(internalloop_cumulative_change),    internalloop_change,{{(C+1),(-1)}},
                                          ST_LENGTH,{{}},false,(-1),(-1),0.0},
{Pair(internalloop_cumulative_asymmetry), internalloop_asymmetry,{{(C+1),(-1)}},
                                          ST_LENGTH,{{}},false,(-1),(-1),0.0},                
#undef Pair
}};
class 
EnergyModel::
FeatureCounter {
public:
  typedef SeqFile SeqT;
  static const string DEFAULT_PARAMS;
  static IndexToParam& index_to_param();
  static string param_name(int idx);
  static int param_index(const string& name);
  static int param_vector_size();
  static ParamGroupType param_group_type(int idx);
  static const Array<int>& offsets(int idx);

  FeatureCounter() : _seq(NULL), _freq(NULL) {
#if !defined(USE_CTENSOR)
    for (int i = 0; i < (int)PARAM_GROUPS.size(); i++) {
      const ParamGroup& gp = PARAM_GROUPS[i];
      tensor(gp.type).set_size0(gp.lengths.begin(), gp.lengths.begin() + gp.rank());
    }
#endif
    init_tensor_values();
  }
  // FeatureCounter(const FeatureCounter& fc) 
  //  : _seq(fc._seq), _data(fc._data) {}
  ~FeatureCounter() {}

  void set_seq(SeqT& seq) {_seq = &seq;}
  int seq(int i) {return (*_seq)[i];}
  void set_freq(Freq& freq) {_freq = &freq;}
  Freq& freq() {return *_freq;}

  void set_default_params() {
    istringstream oss(DEFAULT_PARAMS);
    read_from_file(oss, true);
  }
  void init_tensor_values() {
    for (int i = 0; i < (int)PARAM_GROUPS.size(); i++) {
      const ParamGroup& gp = PARAM_GROUPS[i];
      tensor(gp.type).fill(gp.default_value);
    }
  }

  Array<ValueT> param_vector() {
    Array<ValueT> pv(param_vector_size());
    pv.fill(0.0);
    set_base_from_aux_param();
    for (int i = 0; i < param_vector_size(); i++) {
      const TensorT& tsr = tensor(param_group_type(i));
      const Array<int>& arr = offsets(i);
      ValueT z = 0.0;
      for (int j = 0; j < (int)arr.size(); j++) {
	z += *(tsr.begin() + arr[j]);
      }
      pv[i] = (z / (ValueT)arr.size());
    }
    return pv;
  }
  void clean() {
    Array<ValueT> cv(param_vector_size());
    cv.fill(0.0);
    set_count_vector(cv);
  }
  void set_param_vector(const Array<ValueT>& pv) {
    Check((int)pv.size() == param_vector_size());
    init_tensor_values();
    for (int i = 0; i < param_vector_size(); i++) {
      TensorT& tsr = tensor(param_group_type(i));
      const Array<int>& arr = offsets(i);
      for (int j = 0; j < (int)arr.size(); j++) {
	*(tsr.begin() + arr[j]) = pv[i];
      }
    }
    set_aux_from_base_param();
  }
  Array<ValueT> count_vector() {
    Array<ValueT> cv(param_vector_size());
    cv.fill(0.0);
    set_base_from_aux_count();
    for (int i = 0; i < param_vector_size(); i++) {
      const TensorT& tsr = tensor(param_group_type(i));
      const Array<int>& arr = offsets(i);
      ValueT z = 0.0;
      for (int j = 0; j < (int)arr.size(); j++) {
	z += *(tsr.begin() + arr[j]);
      }
      cv[i] = z;
    }
    return cv;
  }
  void set_count_vector(const Array<ValueT>& cv) {
    Check((int)cv.size() == param_vector_size());
    init_tensor_values();
    for (int i = 0; i < param_vector_size(); i++) {
      TensorT& tsr = tensor(param_group_type(i));
      const Array<int>& arr = offsets(i);
      ValueT factor = 1.0 / (ValueT)arr.size();
      for (int j = 0; j < (int)arr.size(); j++) {
	*(tsr.begin() + arr[j]) = factor * cv[i];
      }
    }
    set_aux_from_base_count();
  }
  void set_aux_from_base_param() {
    for (int i = NPARAMGROUP; i < NPARAMGROUP1; i++) {
      const ParamGroup& gp = PARAM_GROUPS[i];
      TensorT& tsr = tensor(gp.type);
      const TensorT& tsr0 = tensor(gp.pair_group);
      for (int j = 0; j < (int)tsr0.size(); j++) {
	tsr.ref(j) = (tsr0.ref(j) + (j > 0 ? tsr.ref(j-1) : 0));
      }
    }
  }
  void set_base_from_aux_param() {
    for (int i = NPARAMGROUP; i < NPARAMGROUP1; i++) {
      const ParamGroup& gp = PARAM_GROUPS[i];
      const TensorT& tsr = tensor(gp.type);
      TensorT& tsr0 = tensor(gp.pair_group);
      for (int j = 0; j < (int)tsr0.size(); j++) {
	tsr0.ref(j) = (tsr.ref(j) - (j > 0 ? tsr.ref(j-1) : 0));
      }
    }
  }
  void set_aux_from_base_count() {
    for (int i = NPARAMGROUP; i < NPARAMGROUP1; i++) {
      const ParamGroup& gp = PARAM_GROUPS[i];
      TensorT& tsr = tensor(gp.type);
      const TensorT& tsr0 = tensor(gp.pair_group);
      int n = tsr0.size();
      for (int j = (n-1); j >= 0; j--) {
	tsr.ref(j) = (tsr0.ref(j) - (j < (n-1) ? tsr0.ref(j+1) : 0));
      }
    }
  }
  void set_base_from_aux_count() {
    for (int i = NPARAMGROUP; i < NPARAMGROUP1; i++) {
      const ParamGroup& gp = PARAM_GROUPS[i];
      const TensorT& tsr = tensor(gp.type);
      TensorT& tsr0 = tensor(gp.pair_group);
      int n = tsr0.size();
      for (int j = (n-1); j >= 0; j--) {
	tsr0.ref(j) = (tsr.ref(j) + (j < (n-1) ? tsr0.ref(j+1) : 0));
      }
    }
  }
  void read_from_file(istream& fi, bool supress_warning = false) {
    map<string, ValueT> h;
    string buf;
    while (getline(fi, buf)) {
      const Vector<string>& tokens = tokenize<string>(buf);
      if (tokens.empty()) continue;

      Check(tokens.size() == 2);
      h[tokens[0]] = atof(tokens[1].c_str());
    }
    Array<ValueT> pv(param_vector_size());
    pv.fill(0.0);
    for (int i = 0; i < (int)pv.size(); i++) {
      const string& name = param_name(i);
      if (h.find(name) != h.end()) {
	// parameter file is in energy format [kcal/mol]
	pv[i] = energy_to_score(h[name]); 
      } else {
	if (!supress_warning) {
	  Warn("no entry for idx: %d, name: %s", i, name.c_str());
	}
      }
    }
    set_param_vector(pv);
  }
  string param_file_str() {
    const Array<ValueT>& pv = param_vector();
    ostringstream oss;
    for (int i = 0; i < (int)pv.size(); i++) {
      // parameter file is in energy format [kcal/mol]
      oss << param_name(i) << " " << score_to_energy(pv[i]) << "\n";
    }
    return oss.str();
  }
  ValueT e_value(ValueT score) {
    return exp(_kappa_length - _lambda * score);
  }
  ValueT bit_score(ValueT score) {
    return ((_lambda * score - _kappa) / log(2.0));
  }
  void set_e_value_func(int width, 
			ValueT mea_outer_loop_coeff,
			ValueT mea_inner_loop_coeff,
			int length) {
    const pair<ValueT, ValueT>& vals = interpolate_kappa_lambda(width, mea_outer_loop_coeff);
    _kappa = vals.first;
    _kappa_length = (_kappa + (double)log((double)length));
    _lambda = vals.second;
  }
  pair<ValueT, ValueT> interpolate_kappa_lambda(ValueT width, ValueT co) {
    static const CArray<ValueT, 7> width_list = {{
      50, 100, 200, 300, 400, 500, 600
    }};
    static const CArray<ValueT, 8> co_list = {{
      0.1, 0.162725060993692, 0.264794454754009, 0.430886938006377, 0.70116103268473, 1.14096471810023, 1.85663553344511, 3.02121130422912
    }};
    static const CMatrix<ValueT, 7, 8> kappa_table = {{
      -3.58819457274412, -3.72201416447203, -3.93481186533141, -4.26980831214534, -4.80057946450726, -5.55534256952853, -6.75293184797075, -8.36229606167668,
      -3.95191462938873, -4.05179435740328, -4.23063944430025, -4.53526662691379, -5.00504706664435, -5.7515315678853, -6.89411527446374, -8.72893027605953,
      -4.44467163405968, -4.49246791353133, -4.61383945702447, -4.83687928575246, -5.24543749116882, -5.96462336927962, -7.04822075129063, -8.80225529317701,
      -4.74682810200307, -4.78628091204028, -4.86423586996803, -5.02877475600314, -5.3408654349645, -6.01922429228839, -7.09172451612855, -8.93058920403661,
      -4.99604021118035, -4.9962932950187, -5.04193009997818, -5.1675165602067, -5.43540954945008, -6.07102241504439, -7.14316308131162, -8.98531746384682,
      -5.11568950969229, -5.09611175772323, -5.11650404939552, -5.21059125139267, -5.44991251829812, -6.09089901123404, -7.19098591928014, -9.03542529820975,
      -5.23878863858969, -5.20517210399513, -5.14635373638773, -5.24147681597461, -5.47811216179471, -6.12341750363174, -7.21379851941444, -9.14328685655594
    }};
    static const CMatrix<ValueT, 7, 8> lambda_table = {{
      0.192568372091603, 0.208492546583123, 0.235048006584391, 0.28081692216511, 0.359622971755647, 0.52405835781381, 0.876483538854622, 1.19886612787071,
      0.125099236452259, 0.14027874190051, 0.165091307445, 0.207363804883068, 0.287209702892399, 0.452789433506712, 0.817994739560624, 1.08025067705031,
      0.0769119887033333, 0.0912561539049424, 0.114684538402915, 0.161241467971281, 0.256398330027195, 0.437672288105239, 0.825993795781641, 1.12376097554935,
      0.058101135627925, 0.0699480221446484, 0.0932410897294226, 0.142183106696275, 0.247533764995571, 0.435354665238728, 0.842802724129848, 1.0814361957289,
      0.043893208979375, 0.0547518991946947, 0.0756955639252097, 0.122813248430635, 0.235038367306514, 0.435606310030252, 0.836310131472985, 1.09519228764475,
      0.0382778629777011, 0.0491880752380244, 0.0704728486099417, 0.118602920498753, 0.235737732127314, 0.433154273522443, 0.817609677309411, 1.0080172962107,
      0.0339840299119875, 0.0441808580642962, 0.067895308438505, 0.116422139309342, 0.235382692567315, 0.431329919858775, 0.803841064972778, 0.906163999893693
    }}; 
    
    int i = get_neighbor_index(width_list, width);
    int j = get_neighbor_index(co_list, co);
    ValueT s = (width - width_list[i]) / (width_list[i+1] - width_list[i]);
    ValueT t = (co - co_list[j]) / (co_list[j+1] - co_list[j]);
    ValueT sarr[] = {1-s, s};
    ValueT tarr[] = {1-t, t};
    ValueT kappa = 0;
    ValueT lambda = 0;
    for (int u = 0; u < 2; u++) {
      for (int v = 0; v < 2; v++) {
	ValueT fac = (sarr[u] * tarr[v]);
	kappa +=  fac * kappa_table.get(i+u, j+v);
	lambda += fac * lambda_table.get(i+u, j+v);
      }
    }
    return make_pair(kappa, lambda);
  }
  template <typename ArrayT>
  int get_neighbor_index(const ArrayT& zarr, ValueT z) {
    for (int i = 0; i < (int)(zarr.size() - 1); i++) {
      if (z <= zarr[i+1]) {
	return i;
      }
    }
    return (zarr.size() - 2);
  }

#if defined(USE_CTENSOR)
#define TENSOR(name) _tsr_##name
#else
#define TENSOR(name) tensor(name)
#endif
  ValueT score_stack(int i, int j, int n) {
    int c = Alpha::scode(seq(i+1), seq(j));
    int c1 = Alpha::scode(seq(j-1), seq(i+2)); // reverse order
    return TENSOR(stack).ref(c, c1);
  }
  void count_stack(int i, int j, int n, ValueT w) {
    int c = Alpha::scode(seq(i+1), seq(j));
    int c1 = Alpha::scode(seq(j-1), seq(i+2)); // reverse order
    TENSOR(stack).ref(c, c1) += w;
  }
  ValueT score_stem_close(int i, int j, int n) {
    return 0.0;
  }
  void count_stem_close(int i, int j, int n, ValueT w) {
    // no-op
  }
  ValueT score_hairpin(int i, int j) {
    int l = (j-i);
    ValueT sc = 0.0;
    if (l <= C) {
      sc += TENSOR(hairpinloop_cumulative_change).ref(l);
    } else {
      sc += TENSOR(hairpinloop_cumulative_change).ref(C) ;
      sc += log(l / (ValueT)C) * TENSOR(large_loop_factor).ref();
    }
    int c   = Alpha::scode(seq(i), seq(j+1));
    int c1  = Alpha::pcode(seq(i+1), seq(j));
#if true
    if (l <= 4) {
      if (l == 3) {
	sc += TENSOR(triloop).ref(c, c1, seq(i+2));
	sc += TENSOR(terminal_base_pair).ref(c);
      } else {//l == 4
	int c2 = Alpha::pcode(seq(i+2), seq(j-1));
	sc += TENSOR(tetraloop).ref(c, c1, c2);
#ifndef USE_TSTACK4
	sc += TENSOR(tstackh).ref(c, c1);
#else
	sc += TENSOR(tstackh).ref(seq(i), seq(j+1), seq(i+1), seq(j));
#endif
      }
      return sc;
    }
#endif

#ifndef USE_TSTACK4
    sc += TENSOR(tstackh).ref(c, c1);
#else
    sc += TENSOR(tstackh).ref(seq(i), seq(j+1), seq(i+1), seq(j));
#endif
    return sc;
  }
  void count_hairpin(int i, int j, ValueT w) {
    int l = (j-i);
    if (l <= C) {
      TENSOR(hairpinloop_cumulative_change).ref(l) += w;
    } else {
      TENSOR(hairpinloop_cumulative_change).ref(C) += w;
      TENSOR(large_loop_factor).ref() += log(l / (ValueT)C) * w;
    }
    int c   = Alpha::scode(seq(i), seq(j+1));
    int c1  = Alpha::pcode(seq(i+1), seq(j));
#if true
    if (l <= 4) {
      if (l == 3) {
	TENSOR(triloop).ref(c, c1, seq(i+2)) += w;
	TENSOR(terminal_base_pair).ref(c) += w;
      } else {//l == 4
	int c2 = Alpha::pcode(seq(i+2), seq(j-1));
	TENSOR(tetraloop).ref(c, c1, c2) += w;
#ifndef USE_TSTACK4
	TENSOR(tstackh).ref(c, c1) += w;
#else
	TENSOR(tstackh).ref(seq(i), seq(j+1), seq(i+1), seq(j)) += w;
#endif
      }
      return;
    }
#endif
#ifndef USE_TSTACK4
    TENSOR(tstackh).ref(c, c1) += w;
#else
    TENSOR(tstackh).ref(seq(i), seq(j+1), seq(i+1), seq(j)) += w;
#endif
  }
  ValueT score_interior(int i, int j, int ip, int jp) {
    int si  = seq(i);
    int si1 = seq(i+1);
    int sj  = seq(j);
    int sj1 = seq(j+1);
    int sip = seq(ip);
    int sip1= seq(ip+1);
    int sjp = seq(jp);
    int sjp1= seq(jp+1);
    int li = (ip - i);
    int lj = (j - jp);
    int l  = (li + lj);
    int c1 = Alpha::scode(si, sj1);
    int c3 = Alpha::scode(sjp, sip1);
    ValueT sc = 0.0;
    if (li == 0 || lj == 0) {// bulge
      sc += TENSOR(bulgeloop_cumulative_change).ref(l);
      if (l == 1) {
	// no terminal base pair in alifold
	sc += TENSOR(stack).ref(c1, c3);
      } else {
	sc += TENSOR(terminal_base_pair).ref(c1);
	sc += TENSOR(terminal_base_pair).ref(c3);
	// no dangle in alifold
	// if (lj == 0) {
	//   sc += TENSOR(dangle3).ref(c1, si1);
	//   sc += TENSOR(dangle5).ref(c3, sip);
	// } else {
	//   sc += TENSOR(dangle5).ref(c1, sj);
	//   sc += TENSOR(dangle3).ref(c3, sjp1);
	// }
      }
    } else {// internal
      int c2 = Alpha::pcode(si1, sj);
      int c4 = Alpha::pcode(sjp1, sip);
#if true
      if (li <= 2 && lj <= 2) {
	if (l == 2) {
	  sc += TENSOR(int2).ref(c1, si1, sj, c3);
	} else {
	  if (lj == 1) {
	    sc += TENSOR(int21).ref(c1, c2, sip, c3);
	  } else if (li == 1) {
	    sc += TENSOR(int21).ref(c3, c4, sj, c1);
	  } else {
	    sc += TENSOR(int22).ref(c1, c2, c4, c3);
	  }
	}
	return sc;
      }
#endif
      sc += TENSOR(internalloop_cumulative_change).ref(l);
      sc += TENSOR(internalloop_cumulative_asymmetry).ref(abs(li - lj));
#ifndef USE_TSTACK4
      sc += TENSOR(tstacki).ref(c1, c2);
      sc += TENSOR(tstacki).ref(c3, c4);
#else
      sc += TENSOR(tstacki).ref(si, sj1, si1, sj);
      sc += TENSOR(tstacki).ref(sjp, sip1, sjp1, sip);
#endif
    }
    return sc;
  }
  void count_interior(int i, int j, int ip, int jp, ValueT w) {
    int si  = seq(i);
    int si1 = seq(i+1);
    int sj  = seq(j);
    int sj1 = seq(j+1);
    int sip = seq(ip);
    int sip1= seq(ip+1);
    int sjp = seq(jp);
    int sjp1= seq(jp+1);
    int c1 = Alpha::scode(si, sj1);
    int c3 = Alpha::scode(sjp, sip1);
    int li = (ip - i);
    int lj = (j - jp);
    int l  = (li + lj);
    if (li == 0 || lj == 0) {// bulge
      TENSOR(bulgeloop_cumulative_change).ref(l) += w;
      if (l == 1) {// no terminal base pair in alifold
	TENSOR(stack).ref(c1, c3) += w;
      } else {
	TENSOR(terminal_base_pair).ref(c1) += w;
	TENSOR(terminal_base_pair).ref(c3) += w;
	// no dangle in alifold
	// if (lj == 0) {
	//   TENSOR(dangle3).ref(c1, si1) += w;
	//   TENSOR(dangle5).ref(c3, sip) += w;
	// } else {
	//   TENSOR(dangle5).ref(c1, sj) += w;
	//   TENSOR(dangle3).ref(c3, sjp1) += w;
	// }
      }
    } else {// internal
      int c2 = Alpha::pcode(si1, sj);
      int c4 = Alpha::pcode(sjp1, sip);
#if true
      if (li <= 2 && lj <= 2) {
	if (l == 2) {
	  TENSOR(int2).ref(c1, si1, sj, c3) += w;
	} else {
	  if (lj == 1) {
	    TENSOR(int21).ref(c1, c2, sip, c3) += w;
          } else if (li == 1) {
	    TENSOR(int21).ref(c3, c4, sj, c1) += w;
	  } else {
	    TENSOR(int22).ref(c1, c2, c4, c3) += w;
	  }
	}
	return;
      }
#endif
      TENSOR(internalloop_cumulative_change).ref(l) += w;
      TENSOR(internalloop_cumulative_asymmetry).ref(abs(li - lj)) += w;
#ifndef USE_TSTACK4
      TENSOR(tstacki).ref(c1, c2) += w;
      TENSOR(tstacki).ref(c3, c4) += w;
#else
      TENSOR(tstacki).ref(si, sj1, si1, sj) += w;
      TENSOR(tstacki).ref(sjp, sip1, sjp1, sip) += w;
#endif
    }
  }
  ValueT score_multi_close(int i, int j) {
    ValueT sc = TENSOR(multiloop_offset).ref();
#if true
    sc += score_multi_mismatch(i, j);
#endif
    return sc;
  }
  void count_multi_close(int i, int j, ValueT w) {
    TENSOR(multiloop_offset).ref() += w;
#if true
    count_multi_mismatch(i, j, w);
#endif
  }
  ValueT score_multi_open(int i, int j) {
    ValueT sc = TENSOR(multiloop_helix).ref();
    sc += score_multi_mismatch(j, i);
    return sc;
  }
  void count_multi_open(int i, int j, ValueT w) {
    TENSOR(multiloop_helix).ref() += w;
    count_multi_mismatch(j, i, w);
  }
  ValueT score_multi_extend() {
    return TENSOR(multiloop_nuc).ref();
  }
  void count_multi_extend(ValueT w)  {
    TENSOR(multiloop_nuc).ref() += w;
  }
  ValueT score_outer_extend (int i) {
    ScoreT sc = TENSOR(outer_unpaired).ref();
    if (i > 0) {
      int c = seq(i);
      sc += TENSOR(outer_nuc_freq).ref(c);
      if (i > 1) {
	int c1 = Alpha::pcode(seq(i-1), c);
	sc += TENSOR(outer_dinuc_freq).ref(c1);
      }
    }
    return sc;
  }
  void count_outer_extend(int i, ValueT w) {
    TENSOR(outer_unpaired).ref() += w;
    if (i > 0) {
      int c = seq(i);
      TENSOR(outer_nuc_freq).ref(c) += w;
      if (i > 1) {
	int c1 = Alpha::pcode(seq(i-1), c);
	TENSOR(outer_dinuc_freq).ref(c1) += w;
      }
    }
  }
  ValueT score_outer_branch(int i, int j, int i0, int j0) {
    ValueT sc = 0.0;
    int c = Alpha::scode(seq(j), seq(i+1));
#if true
    if (i0 < i) {
      sc += TENSOR(dangle5).ref(c, seq(i));
    }
    if (j < j0) {
      sc += TENSOR(dangle3).ref(c, seq(j+1));
    }
    sc += TENSOR(terminal_base_pair).ref(c);
#else
//     if (i0 < i && j < j0) {
// #ifndef USE_TSTACK4
//       int c1 = Alpha::pcode(seq(j+1), seq(i));
//       sc += TENSOR(tstacke).ref(c, c1);
// #else
//       sc += TENSOR(tstacke).ref(seq(j), seq(i+1), seq(j+1), seq(i));
// #endif
//     }
#endif
    sc += TENSOR(outer_branch).ref();
    // sc += TENSOR(outer_branch_length_factor).ref() * (j-i);
#ifdef USE_TSC_FREQ
    for (int c = 0; c < Alpha::NNCODE; c++) {
      sc += TENSOR(nuc_freq).ref(c) * freq().s[c];
    }
    for (int c = 0; c < Alpha::NPCODE; c++) {
      sc += TENSOR(dinuc_freq).ref(c) * freq().p[c];
    }
#endif
    return sc;
  } 
  void count_outer_branch(int i, int j, int i0, int j0, ValueT w) {
    int c = Alpha::scode(seq(j), seq(i+1));
#if true
    if (i0 < i) {
      TENSOR(dangle5).ref(c, seq(i)) += w;
    }
    if (j < j0) {
      TENSOR(dangle3).ref(c, seq(j+1)) += w;
    }
    TENSOR(terminal_base_pair).ref(c) += w;
#else
//     if (i0 < i && j < j0) {
// #ifndef USE_TSTACK4
//       int c1 = Alpha::pcode(seq(j+1), seq(i));
//       TENSOR(tstacke).ref(c, c1) += w;
// #else
//       TENSOR(tstacke).ref(seq(j), seq(i+1), seq(j+1), seq(i)) += w;
// #endif
//     }
#endif
    TENSOR(outer_branch).ref() += w;
    // TENSOR(outer_branch_length_factor).ref() += (j-i) * w;
#ifdef USE_TSC_FREQ
    for (int c = 0; c < Alpha::NNCODE; c++) {
      TENSOR(nuc_freq).ref(c) += freq().s[c] * w;
    }
    for (int c = 0; c < Alpha::NPCODE; c++) {
      TENSOR(dinuc_freq).ref(c) += freq().p[c] * w;
    }
#endif
  }
  ValueT score_multi_mismatch(int i, int j) {
    int c = Alpha::scode(seq(i), seq(j+1));
    ValueT sc = 0.0;
#if false
#ifndef USE_TSTACK4
    int c1 = Alpha::pcode(seq(i+1), seq(j));
    sc += TENSOR(tstackm).ref(c, c1);
#else
    sc += TENSOR(tstackm).ref(seq(i), seq(j+1), seq(i+1), seq(j));
#endif
#else
    // alifold case
    sc += TENSOR(dangle3).ref(c, seq(i+1));
    sc += TENSOR(dangle5).ref(c, seq(j));
    sc += TENSOR(terminal_base_pair).ref(c);
#endif
    return sc;
  }
  void count_multi_mismatch(int i, int j, ValueT w) {
    int c = Alpha::scode(seq(i), seq(j+1));
#if false
#ifndef USE_TSTACK4
    int c1 = Alpha::pcode(seq(i+1), seq(j));
    TENSOR(tstackm).ref(c, c1) += w;
#else
    TENSOR(tstackm).ref(seq(i), seq(j+1), seq(i+1), seq(j)) += w;
#endif
#else
    // alifold case
    TENSOR(dangle3).ref(c, seq(i+1)) += w;
    TENSOR(dangle5).ref(c, seq(j)) += w;
    TENSOR(terminal_base_pair).ref(c) += w;
#endif
  }
#undef TENSOR

#if !defined(USE_CTENSOR)
  typedef Tensor<ValueT> TensorT;
  typedef CArray<TensorT, NPARAMGROUP1> Data;
  Data _data;
  TensorT& tensor(int type) {return _data[type];}
#else
  typedef TensorBase<ValueT> TensorT;
  CTensor<ValueT> _tsr_large_loop_factor;
  CTensor<ValueT> _tsr_multiloop_offset;
  CTensor<ValueT> _tsr_multiloop_nuc;
  CTensor<ValueT> _tsr_multiloop_helix;
  CTensor<ValueT, (C+1)> _tsr_hairpinloop_change;
  CTensor<ValueT, (C+1)> _tsr_bulgeloop_change;
  CTensor<ValueT, (C+1)> _tsr_internalloop_change;
  CTensor<ValueT, (C+1)> _tsr_internalloop_asymmetry;
  CTensor<ValueT, Alpha::NSCODEN> _tsr_terminal_base_pair;
  CTensor<ValueT, Alpha::NSCODEN, Alpha::NNCODEN> _tsr_dangle3;
  CTensor<ValueT, Alpha::NSCODEN, Alpha::NNCODEN> _tsr_dangle5;
  CTensor<ValueT, Alpha::NSCODEN, Alpha::NSCODEN> _tsr_stack;
#ifndef USE_TSTACK4
  CTensor<ValueT, Alpha::NSCODEN, Alpha::NPCODEN> _tsr_tstackh;
  CTensor<ValueT, Alpha::NSCODEN, Alpha::NPCODEN> _tsr_tstacki;
  CTensor<ValueT, Alpha::NSCODEN, Alpha::NPCODEN> _tsr_tstackm;
  // CTensor<ValueT, Alpha::NSCODEN, Alpha::NPCODEN> _tsr_tstacke;
#else
  CTensor<ValueT, Alpha::NNCODEN, Alpha::NNCODEN, Alpha::NNCODEN, Alpha::NNCODEN> _tsr_tstackh;
  CTensor<ValueT, Alpha::NNCODEN, Alpha::NNCODEN, Alpha::NNCODEN, Alpha::NNCODEN> _tsr_tstacki;
  CTensor<ValueT, Alpha::NNCODEN, Alpha::NNCODEN, Alpha::NNCODEN, Alpha::NNCODEN> _tsr_tstackm;
  // CTensor<ValueT, Alpha::NNCODEN, Alpha::NNCODEN, Alpha::NNCODEN, Alpha::NNCODEN> _tsr_tstacke;
#endif
  CTensor<ValueT, Alpha::NSCODEN, Alpha::NNCODEN, Alpha::NNCODEN, Alpha::NSCODEN> _tsr_int2;
  CTensor<ValueT, Alpha::NSCODEN, Alpha::NPCODEN, Alpha::NNCODEN, Alpha::NSCODEN> _tsr_int21;
  CTensor<ValueT, Alpha::NSCODEN, Alpha::NPCODEN, Alpha::NPCODEN, Alpha::NSCODEN> _tsr_int22;
  CTensor<ValueT, Alpha::NSCODEN, Alpha::NPCODEN, Alpha::NNCODEN> _tsr_triloop;
  CTensor<ValueT, Alpha::NSCODEN, Alpha::NPCODEN, Alpha::NPCODEN> _tsr_tetraloop;
  CTensor<ValueT> _tsr_outer_unpaired;
  CTensor<ValueT> _tsr_outer_branch;
  // CTensor<ValueT> _tsr_outer_branch_length_factor;
#ifdef USE_TSC_FREQ
  CTensor<ValueT, Alpha::NNCODEN> _tsr_nuc_freq;
  CTensor<ValueT, Alpha::NPCODEN> _tsr_dinuc_freq;
#endif
  CTensor<ValueT, Alpha::NNCODEN> _tsr_outer_nuc_freq;
  CTensor<ValueT, Alpha::NPCODEN> _tsr_outer_dinuc_freq;

  CTensor<ValueT, (C+1)> _tsr_hairpinloop_cumulative_change;
  CTensor<ValueT, (C+1)> _tsr_bulgeloop_cumulative_change;
  CTensor<ValueT, (C+1)> _tsr_internalloop_cumulative_change;
  CTensor<ValueT, (C+1)> _tsr_internalloop_cumulative_asymmetry;
  TensorT& tensor(int type) {
    static CTensor<ValueT> tsr_null;
    switch (type) {
#define Item(name) case name: return _tsr_##name
      Item(large_loop_factor);
      Item(multiloop_offset);
      Item(multiloop_nuc);
      Item(multiloop_helix);
      Item(hairpinloop_change);
      Item(bulgeloop_change);
      Item(internalloop_change);
      Item(internalloop_asymmetry);
      Item(terminal_base_pair);
      Item(dangle3);
      Item(dangle5);
      Item(stack);
      Item(tstackh);
      Item(tstacki);
      Item(tstackm);
      // Item(tstacke);
      Item(int2);
      Item(int21);
      Item(int22);
      Item(triloop);
      Item(tetraloop);

      Item(outer_unpaired);
      Item(outer_branch);
      // Item(outer_branch_length_factor);
#ifdef USE_TSC_FREQ
      Item(nuc_freq);
      Item(dinuc_freq);
#endif
      Item(outer_nuc_freq);
      Item(outer_dinuc_freq);

      Item(hairpinloop_cumulative_change);
      Item(bulgeloop_cumulative_change);
      Item(internalloop_cumulative_change);
      Item(internalloop_cumulative_asymmetry);
    default: Die("bad tensor type %d", type); return tsr_null;
#undef Item
    }
  }
#endif

private:
  SeqT* _seq;
  Freq* _freq;
  ValueT _kappa;
  ValueT _kappa_length;
  ValueT _lambda;
  FeatureCounter(const FeatureCounter& other);
};
#include "./vienna/energy_model_default_params.hpp"
class EnergyModel::IndexToParam {
public:
  typedef Vector<pair<ParamGroupType, Array<int> > > Data;
  IndexToParam() {
    for (int i = 0; i < (int)NPARAMGROUP; i++) {// not PARAM_GROUP.size();
      const ParamGroup& gp = PARAM_GROUPS[i];
      const FeatureCounter::TensorT& tsr = _fc.tensor(gp.type);
      for (int j = 0; j < (int)tsr.size(); j++) {
	const Array<int>& idxs = tsr.get_indexes(j);
	bool new_param = true;
	switch (gp.subscript_type) {
	case ST_NONE:
	  break;
	case ST_ALPHA:
	  for (int k = 0; k < (int)idxs.size(); k++) {
	    int idx = idxs[k];
	    switch (gp.alpha_types[k]) {
	    case AT_NONE:
	      Die("bad alpha type %d", gp.alpha_types[k]);
	      break;
	    case AT_NCODE:
	      if (idx >= Alpha::N_N) {
		new_param = false;
	      }
	      break;
	    case AT_PCODE:
	      if (idx >= Alpha::P_NN) {
		new_param = false;
	      }
	      break;
	    case AT_SCODE:
	      if (idx >= Alpha::S_MM) {
		new_param = false;
	      }
	      break;
	    default:
	      Die("bad alpha type %d", gp.alpha_types[k]);
	      new_param = false;
	      break;
	    }
	    if (!new_param) break;
	  }
	  break;
	case ST_LENGTH:
	  if (gp.base_subscript >= 0) {
	    for (int k = 0; k < (int)idxs.size(); k++) {
	      if (idxs[k] < gp.base_subscript) {
		new_param = false;
		break;
	      }
	    }
	  }
	  if (gp.sum_bound >= 0) {
	    if (gp.sum_bound < accumulate(idxs.begin(), idxs.end(), 0)) {
	      new_param = false;
	    }
	  }
	  break;
	default:
	  Die("bad subscript type %d", gp.subscript_type);
	  break;
	}
	Array<int> idxs1(idxs);
	reverse(idxs1.begin(), idxs1.end());
	if (gp.symmetric) {
	  // idxs1 < idxs
	  if (lexicographical_compare(idxs1.begin(), idxs1.end(),
				      idxs.begin(), idxs.end())) {
	    new_param = false;
	  }
	}
	if (!new_param) continue;
	
	Vector<int> arr(1);
	arr[0] = j;
	// idxs < idxs1
	bool b = lexicographical_compare(idxs.begin(), idxs.end(),
					 idxs1.begin(), idxs1.end());
	if (gp.symmetric && b) {
	  int j1 = tsr.get_index(idxs1);
	  arr.push_back(j1);
	}
	_data.push_back(make_pair(gp.type, Array<int>(arr.begin(), arr.end())));
      }
    }
  }
  int size() const {return _data.size();}
  Data::const_reference operator[](int idx) const {return _data[idx];}
  string param_name(int idx) {
    const ParamGroup& gp = PARAM_GROUPS[_data[idx].first];
    const Array<int>& arr = _data[idx].second;
    const FeatureCounter::TensorT& tsr = _fc.tensor(gp.type);
    const Array<int>& idxs = tsr.get_indexes(arr[0]);
    ostringstream oss;
    oss << gp.name;
    switch (gp.subscript_type) {
    case ST_NONE:
      break;
    case ST_ALPHA:
      oss << "_";
      for (int k = 0; k < (int)idxs.size(); k++) {
	switch (gp.alpha_types[k]) {
	case AT_NONE:
	  Die("bad alpha type %d", gp.alpha_types[k]);
	  break;
	case AT_NCODE:
	  oss << Alpha::ncode_to_str(idxs[k]);
	  break;
	case AT_PCODE:
	  oss << Alpha::pcode_to_str(idxs[k]);
	  break;
	case AT_SCODE:
	  oss << Alpha::scode_to_str(idxs[k]);
	  break;
	default:
	  Die("bad alpha type %d", gp.alpha_types[k]);
	  break;
	}
      }
      break;
    case ST_LENGTH:
      for (int k = 0; k < (int)idxs.size(); k++) {
	oss << "_" << idxs[k];
      }
      break;
    default:
      Die("bad subscript type %d", gp.subscript_type);
      break;
    }
    return oss.str();
  }
  int param_index(const string& name) {
    for (int i = 0; i < (int)_data.size(); i++) {
      if (param_name(i) == name) return i;
    }
    return (-1);
  }
  int param_vector_size() const {return _data.size();}
private:
  Data _data;
  FeatureCounter _fc;
};
inline EnergyModel::IndexToParam& 
EnergyModel::FeatureCounter::
index_to_param() {
  static IndexToParam _index_to_param;
  return _index_to_param;
}
inline string 
EnergyModel::FeatureCounter::
param_name(int idx) {return index_to_param().param_name(idx);}
inline int 
EnergyModel::FeatureCounter::
param_index(const string& name) {return index_to_param().param_index(name);}
inline int 
EnergyModel::FeatureCounter::
param_vector_size() {return index_to_param().param_vector_size();}
inline EnergyModel::ParamGroupType 
EnergyModel::FeatureCounter::
param_group_type(int idx) {return index_to_param()[idx].first;}
inline const Array<int>& 
EnergyModel::FeatureCounter::
offsets(int idx) {return index_to_param()[idx].second;}
}

#endif
