/*
 * opt.cpp
 */
#include "./util.hpp"
#include "./opt.hpp"

namespace RFOLD {
Opt::EntryData::
EntryData()
  : name(), type(NO_ARGUMENT), dflt(), value(), val_is_set(false) {}

Opt::EntryData::
EntryData(const Entry& entry)
  : name(entry.name), type(entry.type), dflt(entry.dflt), value(),
    val_is_set(false)
{
}
Opt::EntryData::
EntryData(const EntryData& entry_data)
 : name(entry_data.name), type(entry_data.type), dflt(entry_data.dflt),
   value(entry_data.value), val_is_set(entry_data.val_is_set)
{
}
Opt::EntryData::
~EntryData()
{
}
Opt::EntryData&
Opt::EntryData::
operator=(const EntryData& entry_data)
{
  if (this == &entry_data) return *this;

  name = entry_data.name;
  type = entry_data.type;
  dflt = entry_data.dflt;
  value = entry_data.value;
  val_is_set = entry_data.val_is_set;
  return *this;
}
void
Opt::EntryData::
set_value(const string& value0)
{
  if (value0.size() >= 2
      && ((value0[0] == '"' && value0[value0.size() - 1] == '"')
	  || (value0[0] == '\'' && value0[value0.size() - 1] == '\''))
      ) {
    value = value0.substr(1, value0.size() - 2);
  } else {	    
    value = value0;
  }
  val_is_set = true;
}
const string&
Opt::EntryData::
get_value() const
{
  return (val_is_set ? value : dflt); 
}
string 
Opt::EntryData::
to_s() const
{
  ostringstream oss;
  oss << "{ name: " << name << ", ";
  oss << "dflt: " << dflt << ", ";
  oss << "value: " << value << ", ";
  oss << "val_is_set: " << val_is_set << " }";
  return oss.str();
}
void
Opt::EntryData::
print() const
{
  cout << to_s() << flush;
}
Opt::
Opt(const Entry* const opt_table) : fname(), table()
{
  for (int i = 0; opt_table[i].name != NULL; i++) {
    const Entry& entry = opt_table[i];
    Assert(table.find(entry.name) == table.end(),
	   "duplicated entry name %s", entry.name);
    table.insert(make_pair(string(entry.name), EntryData(entry)));
  }
}
Opt::
~Opt()
{
}
void
Opt::
set_from_file(const string& filename)
{
  ifstream fi(filename.c_str());
  string buf;

  if (!fi) {
    Warn("cannot open paramfile=%s\n", filename.c_str());
    return;
  }
  fname = filename;

  while (getline(fi, buf)) {
    StringTokenizer st(buf);
    string key = st.next_token();
    string val = st.next_token();
    if (key.empty() 
	|| key[0] == '#' 
	|| key.find("---") == 0
	|| key == "options:"
	|| key == "fname:") continue;
    
    bool sep_found = false;
    string::size_type pos = key.size() - 1;
    if (key[pos] == ':') {
      sep_found = true;
      key.erase(pos);
    }
    if (!has_key(key)) {
      Die("unrecognized parameter %s in file %s",
	  key.c_str(), filename.c_str());
    }
    if (!sep_found && val == ":") {
      sep_found = true;
      val = st.next_token();
    }
    if (!sep_found) Die("separator ':' not found in line\n  %s", 
		       buf.c_str());

    table[key].set_value(val);
  }
}
void
Opt::
set_value(const string& key, const string& value)
{
  Check(has_key(key));
  table[key].set_value(value);
}   
void
Opt::
set_from_args(int& argc, char**& argv)
{
  static const string param_file = "paramfile";
  int i = 1;
  while ((i < argc) && (argv[i][0] == '-')) {
    const string& str = string(argv[i] + (argv[i][1] == '-' ? 2 : 1));
    bool has_equal = (str.find('=') != string::npos);
    StringTokenizer st(str);
    const string& key = st.next_token("=");
    string val = st.next_token();

    if (!has_key(key) && key != param_file) {
      Die("unrecognized option %s", key.c_str());
    }
    i++;
    
    if (val.empty()) {
      if (table[key].required_argument() || key == param_file) {
	if (has_equal || (i >= argc) || (argv[i][0] == '-')) {
	  Die("no argument for parameter %s", key.c_str());
	} else {
	  val = argv[i];
	  i++;
	}
      } else {
	val = (has_equal ? "false" : "true");
      }
    }

    if (key == param_file) {
      set_from_file(val);
    } else {
      table[key].set_value(val);
    }
  }

  for (int j = i; j < argc; j++) {
    if (argv[j][0] == '-') {
      Die("parsing stopped at argv[%d] = %s", i, argv[i]);
    }
  }

  argc -= i;
  argv += i;
  return;
}
bool
Opt::
has_key(const string& name) const
{
  return (table.find(name) != table.end());
}
const string&
Opt::
get_str(const string& name) const
{
  if (!has_key(name)) Die("unrecognized option %s", name.c_str());

  return table.find(name)->second.get_value();
}
int
Opt::
get_int(const string& name) const
{
  if (!has_key(name)) Die("unrecognized option %s", name.c_str());

  return atoi(table.find(name)->second.get_value().c_str());
}
double
Opt::
get_dbl(const string& name) const
{
  if (!has_key(name)) Die("unrecognized option %s", name.c_str());

  return atof(table.find(name)->second.get_value().c_str());
}
bool
Opt::
get_bool(const string& name) const
{
  if (!has_key(name)) Die("unrecognized option %s", name.c_str());

  static const string true_values[] = {
    "true", "1", "yes", "on", "t", "y", ""
  };

  const string& value = table.find(name)->second.get_value();
  for (const string* p = true_values; !p->empty(); p++) {
    if (value == *p) return true;
  }
  return false;
}
void
Opt::
check_int_lower_bound(const string& name, int min_value) const
{
  if (get_int(name) < min_value)
    Die("%s < %d is not a valid option", name.c_str(), min_value);
}
void
Opt::
check_int_upper_bound(const string& name, int max_value) const
{
  if (get_int(name) > max_value)
    Die("%s > %d is not a valid option", name.c_str(), max_value);
}
void
Opt::
check_int_range(const string& name, int min_value, int max_value) const
{
  check_int_lower_bound(name, min_value);
  check_int_upper_bound(name, max_value);
}
void
Opt::
check_dbl_lower_bound(const string& name, double min_value) const
{
  if (get_dbl(name) < min_value)
    Die("%s < %lf is not a valid option", name.c_str(), min_value);
}
void
Opt::
check_dbl_upper_bound(const string& name, double max_value) const
{
  if (get_dbl(name) > max_value)
    Die("%s > %lf is not a valid option", name.c_str(), max_value);
}
void
Opt::
check_dbl_range(const string& name, double min_value, double max_value) const
{
  check_dbl_lower_bound(name, min_value);
  check_dbl_upper_bound(name, max_value);
}
void
Opt::
check_str_entry(const string& name, const string& entries) const
{
  StringTokenizer st(entries);
  const string& value = get_str(name);
  while (st.has_more_tokens()) {
    if (value == st.next_token()) return;
  }
  Die("%s=%s is not a valid option", name.c_str(), value.c_str());
}
void
Opt::
print_header(ostream& fo) const
{
  for (Table::const_iterator ti = table.begin(); ti != table.end(); ti++) {
    const string& key = ti->first;
    const string& value = ti->second.get_value();
    fo << "#" << key << "\t" << value << endl;
  }
}
void
Opt::
print_yaml(ostream& fo) const
{
  const string& indent = "  ";
  fo << "---\noptions:\n";
  fo << indent << "fname: " << fname << "\n";
  for (Table::const_iterator ti = table.begin(); ti != table.end(); ti++) {
    const EntryData& entry_data = ti->second;
    if (entry_data.value_is_set()) {
      fo << indent << entry_data.get_name() << ": ";
      fo << entry_data.get_value() << "\n";
    }
  }
  fo << flush;
}
void
Opt::
set_from_yaml(istream& fi)
{
  string buf;
  while (getline(fi, buf)) {
    StringTokenizer st(buf);
    while (st.has_more_tokens()) {
      const string& tok = st.next_token();
      if (tok == "---" || tok == "options:") continue;
      if (tok == "fname:") {
	fname = st.next_token();
	continue;
      }

      const string& name = tok.substr(0, tok.size() - 1);
      if (has_key(name)) {
	table[name].set_value(st.next_token());
      } else {
	Die("unrecognized option %s", name.c_str());
      }
    }
  }
}
string
Opt::
to_s() const
{
  ostringstream oss;
  oss << "fname: " << fname << "\n";
  oss << "nopt: " << table.size() << "\n";
  for (Table::const_iterator ti = table.begin(); ti != table.end(); ti++) {
    const string& key = ti->first;
    const string& value = ti->second.to_s();
    oss << key << ": " << value << "\n";
  }
  return oss.str();
}
void
Opt::
print() const 
{
  cout << to_s() << flush;
}
}  
#if false
// check behavior
Opt::Entry opt_table[] = {
  {"leneage",        Opt::REQUIRED_ARGUMENT, "WAVG"},
  {"idxbit",         Opt::REQUIRED_ARGUMENT, "2"},
  {"wdsiz",          Opt::REQUIRED_ARGUMENT, "3"},
  {"lensft",         Opt::REQUIRED_ARGUMENT, "1"},
  {"lenbit",         Opt::REQUIRED_ARGUMENT, "8"},
  {"block_size",     Opt::REQUIRED_ARGUMENT, "10"},
  {"init_size",      Opt::REQUIRED_ARGUMENT, "20"},
  {"gap_open_core",   Opt::REQUIRED_ARGUMENT, "-5.0"},
  {"gap_extend_score", Opt::REQUIRED_ARGUMENT, "-1.0"},
  {"lambda",         Opt::REQUIRED_ARGUMENT, "5.0"},
  {"smtxfile",       Opt::REQUIRED_ARGUMENT, "RIBOSUM45.mat"},
  {"seqfile",        Opt::REQUIRED_ARGUMENT, "seqdata1.txt"},
  {"outfile",        Opt::REQUIRED_ARGUMENT, "outfile.txt"},
  {"refile_tree_max_iter", Opt::REQUIRED_ARGUMENT, "3"},
  {"flag",           Opt::NO_ARGUMENT, NULL},
  {NULL,             Opt::NO_ARGUMENT,       NULL}
};

int
main()
{
  Opt opt(opt_table);
  ostream& fo = cout;
  char* args[] = {
    "main", "-idxbit", "1000", "-seqfile=data.txt", "file"
  };
  int argc = sizeof(args) / sizeof(char*);
  char** argv = args;

  opt.print(fo);
  opt.set_from_file("../data/params.txt");
  opt.print(fo);
  opt.set_from_args(argc, argv);
  opt.print(fo);
  fo << "remaining arguments:" << argc << "\n";
  for (int i = 0; i < argc; i++) {
    fo << argv[i] << endl;
  }
  fo << "size:" << opt.get_int("size") << "\n";
  fo << "double:" << opt.get_dbl("lambda") << "\n";
  fo << "flag:" << opt.get_bool("flag") << "\n";
  fo << "outfile:" << opt.get_str("outfile") << endl;
  return 0;
}

#endif
